// server/domains/pricing.ts
import { applyMarkup, extractTld } from '../lib/markup';

type TldPricing = {
  wholesaleCents: number;  // wholesale cost from registrar
  renewCents?: number;     // optional, for later
  transferCents?: number;  // optional
  currency?: string;       // default USD
};

// Wholesale pricing table (SuperNova's actual OpenSRS wholesale costs)
const WHOLESALE_TABLE: Record<string, TldPricing> = {
  com: { wholesaleCents: 1450 },  // $14.50 wholesale → $18.99 retail (vs GoDaddy $20.99)
  net: { wholesaleCents: 1999 },  // $19.99 wholesale → $20.99 retail (vs GoDaddy $22.99)
  org: { wholesaleCents: 1499 },  // $14.99 wholesale → $21.49 retail (vs GoDaddy $23.99)
  co:  { wholesaleCents: 1499 },  // $14.99 wholesale → $26.99 retail (vs GoDaddy $29.99)
  io:  { wholesaleCents: 4500 },  // $45.00 wholesale → maintain current .io pricing
  ai:  { wholesaleCents: 16950 }, // $169.50 wholesale → $179.99 retail (vs GoDaddy $199.99)
};

export interface DomainPricing {
  wholesale: {
    priceCents: number;
    currency: string;
  };
  retail: {
    priceCents: number;
    currency: string;
  };
}

// Get comprehensive pricing for domain (wholesale + retail with markup)
export function getDomainPricing(domain: string): DomainPricing {
  const tld = domain.split(".").pop()?.toLowerCase() || "com";
  const row = WHOLESALE_TABLE[tld];
  const wholesaleCents = row?.wholesaleCents || 1699; // fallback $16.99 wholesale
  const currency = row?.currency || "USD";

  // Convert to dollars for markup calculation  
  const wholesaleDollars = wholesaleCents / 100;
  const tldWithDot = extractTld(domain);
  
  // Apply markup to get retail price
  const retailDollars = applyMarkup({ 
    wholesale: wholesaleDollars, 
    tld: tldWithDot 
  });
  const retailCents = Math.round(retailDollars * 100);

  return {
    wholesale: {
      priceCents: wholesaleCents,
      currency
    },
    retail: {
      priceCents: retailCents,
      currency
    }
  };
}

// Legacy function for backward compatibility - returns retail pricing
export function getRetailPriceForDomain(domain: string) {
  const pricing = getDomainPricing(domain);
  return { 
    priceCents: pricing.retail.priceCents, 
    currency: pricing.retail.currency 
  };
}