import OpenAI from "openai";

// the newest OpenAI model is "gpt-5" which was released August 7, 2025. do not change this unless explicitly requested by the user

export interface GenerateSlogansRequest {
  brandName: string;
  description?: string;
  tone?: string;
  industry?: string;
  audience?: string;
  maxResults?: number;
}

export interface SloganResponse {
  suggestions: string[];
}

const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
});

// Tone-specific seed prompts for Nova (GPT-4) to produce the right vibe per tone
const TONE_SEED_PROMPTS = {
  Professional: `Generate concise, confident, and trustworthy slogans. 
Emphasize reliability, expertise, and authority. 
Avoid humor or slang. Keep it polished and businesslike.`,

  Playful: `Generate lighthearted, witty, and fun slogans. 
Use casual language, puns, or humor where natural. 
Make it feel approachable and energetic.`,

  Bold: `Generate strong, daring, and attention-grabbing slogans. 
Short punchy words. Impactful tone. 
Make the brand sound fearless and memorable.`,

  Minimal: `Generate clean, simple, and stripped-down slogans. 
Prefer 2–4 words max. No filler. 
Focus on clarity and elegance.`,

  Luxury: `Generate elegant, premium-sounding slogans. 
Emphasize sophistication, exclusivity, and timeless appeal. 
Avoid slang, keep the tone upscale and refined.`,

  Friendly: `Generate warm, approachable, and inviting slogans. 
Use inclusive and encouraging language. 
Make the brand feel like a helpful friend.`
};

// Helper function to get the system message with tone-specific prompt
function getSystemMessage(tone: string): string {
  const baseSysMessage = "You are Nova, an expert brand strategist. Generate short, memorable taglines with punch. Avoid clichés and trademarked phrases.";
  const toneSeedPrompt = TONE_SEED_PROMPTS[tone as keyof typeof TONE_SEED_PROMPTS];
  
  if (toneSeedPrompt) {
    return `${baseSysMessage}\n\n${toneSeedPrompt}`;
  }
  
  return baseSysMessage;
}

// Helper function to parse plain-text slogan response
function parsePlainTextSlogans(content: string, maxResults: number, brandName: string): SloganResponse {
  // Split by lines and clean up
  let lines = content
    .split('\n')
    .map(line => line.trim())
    .filter(line => line.length > 0)
    .filter(line => !/^[^\w\s]*$/.test(line)) // Remove lines with only punctuation
    .map(line => {
      // Remove leading numbers, bullets, dashes, quotes
      line = line.replace(/^[\d\.\)\-\*\+•>\s]*/, '').trim();
      // Strip surrounding quotes
      line = line.replace(/^["'`]|["'`]$/g, '').trim();
      return line;
    })
    .filter(line => line.length > 0);

  // Deduplication via case-insensitive compare
  const uniqueSuggestions: string[] = [];
  const seenLowercase = new Set<string>();

  for (const slogan of lines) {
    const lowerSlogan = slogan.toLowerCase();
    if (!seenLowercase.has(lowerSlogan)) {
      seenLowercase.add(lowerSlogan);
      uniqueSuggestions.push(slogan);
      
      if (uniqueSuggestions.length >= maxResults) {
        break;
      }
    }
  }

  // If we don't have enough unique suggestions, pad with fallbacks
  while (uniqueSuggestions.length < maxResults) {
    const fallbacks = [
      `${brandName} - Excellence Redefined`,
      `Your Success, Our Mission`,
      `Innovation That Inspires`,
      `Quality You Can Trust`,
      `${brandName} - Leading the Way`,
      `Experience the Difference`,
      `Where Quality Meets Innovation`,
      `Your Partner in Success`,
      `${brandName} - Beyond Expectations`,
      `Crafted for Excellence`,
      `Your Vision, Our Expertise`,
      `${brandName} - Trusted Choice`
    ];

    for (const fallback of fallbacks) {
      const lowerFallback = fallback.toLowerCase();
      if (!seenLowercase.has(lowerFallback) && uniqueSuggestions.length < maxResults) {
        seenLowercase.add(lowerFallback);
        uniqueSuggestions.push(fallback);
      }
    }
    
    // Prevent infinite loop
    if (uniqueSuggestions.length === 0) {
      uniqueSuggestions.push(`${brandName} - Excellence Redefined`);
      break;
    }
  }

  return { suggestions: uniqueSuggestions };
}

export async function generateSlogans(request: GenerateSlogansRequest): Promise<SloganResponse> {
  const { 
    brandName, 
    description = "", 
    tone = "Professional", 
    industry = "", 
    audience = "", 
    maxResults = 8 
  } = request;

  // Clamp maxResults to valid range [1, 12]
  const clampedMaxResults = Math.max(1, Math.min(12, maxResults));

  // EXACT prompt template as specified
  const prompt = `Brand name: ${brandName}
Description: ${description || "Not specified"}
Tone: ${tone}
Industry: ${industry || "Not specified"}
Audience: ${audience || "Not specified"}

Generate ${clampedMaxResults} memorable taglines. Return as a plain-text list, one per line.`;

  // Try GPT-5 first, fallback to gpt-4o-mini
  let response: SloganResponse;
  
  try {
    // First attempt with GPT-5
    const completion = await openai.chat.completions.create({
      model: "gpt-5",
      messages: [
        {
          role: "system",
          content: getSystemMessage(tone)
        },
        {
          role: "user",
          content: prompt
        }
      ],
      temperature: 0.8,
      max_tokens: 400
    });

    const content = completion.choices[0]?.message?.content;
    if (!content) {
      throw new Error("No response from OpenAI");
    }

    // Parse plain-text response
    response = parsePlainTextSlogans(content, clampedMaxResults, brandName);

  } catch (error) {
    console.log("GPT-5 failed, falling back to gpt-4o-mini:", error);
    
    try {
      // Fallback to gpt-4o-mini
      const completion = await openai.chat.completions.create({
        model: "gpt-4o-mini",
        messages: [
          {
            role: "system",
            content: getSystemMessage(tone)
          },
          {
            role: "user",
            content: prompt
          }
        ],
        temperature: 0.8,
        max_tokens: 400
      });

      const content = completion.choices[0]?.message?.content;
      if (!content) {
        throw new Error("No response from OpenAI fallback");
      }

      // Parse plain-text response
      response = parsePlainTextSlogans(content, clampedMaxResults, brandName);

    } catch (fallbackError) {
      console.error("Both GPT-5 and gpt-4o-mini failed:", fallbackError);
      // Return fallback slogans if both AI models fail
      response = {
        suggestions: [
          `${brandName} - Excellence Redefined`,
          `Your Success, Our Mission`,
          `Innovation That Inspires`,
          `Quality You Can Trust`,
          `${brandName} - Leading the Way`,
          `Experience the Difference`,
          `Where Quality Meets Innovation`,
          `Your Partner in Success`
        ].slice(0, clampedMaxResults)
      };
    }
  }

  return response;
}