import OpenAI from "openai";
import type { AiJob, AiResponse, AiConfig } from "@shared/ai-types";

// the newest OpenAI model is "gpt-5" which was released August 7, 2025. do not change this unless explicitly requested by the user

const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
});

// Default AI configuration
const DEFAULT_CONFIG: AiConfig = {
  model: "gpt-5",
  maxTokens: 2000,
  temperature: 0.7,
};

// Tone-specific system prompts for different writing styles
const TONE_SYSTEM_PROMPTS = {
  Professional: `You are a professional business consultant and writer. Write in a polished, confident, and authoritative tone. Use business terminology appropriately and maintain a formal, credible style throughout. Be comprehensive but concise.`,

  Friendly: `You are a friendly business mentor and advisor. Write in a warm, encouraging, and conversational tone while maintaining professionalism. Use inclusive language that makes the reader feel supported and motivated. Explain concepts clearly and avoid unnecessary jargon.`,

  Bold: `You are a dynamic business strategist and thought leader. Write with confidence, conviction, and impact. Use strong, decisive language that inspires action. Be direct and ambitious in your recommendations. Emphasize competitive advantages and growth opportunities.`,

  Minimal: `You are a minimalist business writer who values clarity and brevity. Write in a clean, simple, and direct style. Use short sentences and paragraphs. Focus on essential information only. Avoid redundancy and filler words.`,
};

// Section-specific base prompts per Nova's exact specifications
const SECTION_BASE_PROMPTS = {
  'executive-summary': (company?: string) => `Write a concise, investor-ready executive summary for ${company || '{{company}}'} using: business model, offerings (full suite), target market, competitive edge, and financialHeadline (Launch/Growth/Scale phrasing if present). No placeholders.`,
  
  'company-overview': () => `Mission, vision, value prop, origin, legal structure/location, current stage — factual and compact.`,
  
  'products-services': () => `3–7 bullets (1 line each) + 1-paragraph value prop. If tiers/pricing exist, summarize in ≤3 lines.`,
  
  'market-analysis': () => `ICP, 3–4 trends, 2–4 competitors with one-line differentiation, and a mini-SWOT (one line per S/W/O/T). Keep TAM/SAM/SOM qualitative if unknown.`,
  
  'marketing-sales': () => `Positioning, 3–6 channels, TOFU→MOFU→BOFU in one line, retention tactics, and 4–6 KPIs (CAC, LTV, MRR, churn).`,
  
  'operations-plan': () => `Tech/infra, processes (support, releases, QA, billing), key vendors/tools, SLAs/security, and 3–5 ops KPIs.`,
  
  'org-management': () => `Leadership roles; include 'Strategic AI Partner: Nova AI — Co-Founder / Operations & Data Intelligence' if relevant; hiring plan; optional advisors.`,
  
  'financial-plan': (year0?: string) => `Revenue streams, assumptions, forecast using Launch (Q4 year0), Growth (year1), Scale (year2). Mention break-even if implied; add 2–3 risks + mitigations. Use financialHeadline if present; no invented numbers.`,
  
  'milestones-roadmap': () => `4–8 bullets grouped by phase: Launch, Growth, Scale, Beyond. Each bullet starts with a verb and includes an outcome metric if present.`,
  
  'future-expansion': () => `2–4 expansion vectors (integrations, adjacent products, partnerships/channels, long-term vision) tied back to model/market.`,
  
  'partnerships-note': () => `Rationale, partner types, criteria, engagement model (referrals, rev-share, white-label), and the customer benefit — brief, practical.`,
  
  'exit-strategy': () => `Primary path (typical acquirers), alternatives (IPO/private growth), timing triggers, and value narrative — neutral if signals are light.`,
  
  'custom': () => `Provide relevant business plan content based on the section title and context provided.`,
};

// Action-specific prompts
const ACTION_PROMPTS = {
  generate: (job: AiJob) => {
    // [SYSTEM] Fixed instruction per SN's exact spec
    const systemPrompt = `You are drafting a professional business plan section. Be accurate, clear, and business-appropriate. No bracket placeholders like [Company Name]. Use the provided context. If data is missing, phrase it neutrally.`;
    
    // [CONTEXT] Business Brief
    const brief = job.context?.businessBrief;
    let briefContext = '';
    if (brief) {
      const briefParts = [];
      if (brief.company) briefParts.push(`Company: ${brief.company}`);
      if (brief.industry) briefParts.push(`Industry: ${brief.industry}`);
      if (brief.businessModel) briefParts.push(`Business Model: ${brief.businessModel}`);
      if (brief.productsServices) briefParts.push(`Products/Services: ${brief.productsServices}`);
      if (brief.offeringsFull) briefParts.push(`Offerings (Full Suite): ${brief.offeringsFull}`);
      if (brief.market) briefParts.push(`Target Market: ${brief.market}`);
      if (brief.differentiator) briefParts.push(`Competitive Edge: ${brief.differentiator}`);
      if (brief.financialHeadline) briefParts.push(`Financial Headline: ${brief.financialHeadline}`);
      
      if (briefParts.length > 0) {
        briefContext = `\n\nBusiness Brief:\n${briefParts.join('\n')}`;
      }
    }
    
    // [CONTEXT] Section ID
    const sectionContext = `\nSection: ${job.sectionKind}`;
    
    // [TASK] Section Base Prompt
    const basePromptFn = SECTION_BASE_PROMPTS[job.sectionKind as keyof typeof SECTION_BASE_PROMPTS] || SECTION_BASE_PROMPTS.custom;
    const company = brief?.company || job.context?.businessName;
    const year0 = brief?.financialHeadline?.match(/20\d{2}/)?.[0] || new Date().getFullYear().toString();
    const basePrompt = typeof basePromptFn === 'function' ? basePromptFn(job.sectionKind === 'executive-summary' ? company : job.sectionKind === 'financial-plan' ? year0 : undefined) : basePromptFn;
    const taskContext = `\n\nTask: ${basePrompt}`;
    
    // [USER] Optional Prompt (from drawer field, if given)
    const userPromptContext = job.userPrompt && job.userPrompt.trim().length > 0
      ? `\n\nUser Instructions:\n${job.userPrompt}`
      : '';
    
    // [USER] Editor Content (if present, used as context)
    const editorContext = job.existingContent && job.existingContent.trim().length > 0
      ? `\n\nExisting Draft Context:\n${job.existingContent}`
      : '';
    
    // [STYLE] Length and Tone - SN's exact word count targets
    const length = job.length || 'Standard';
    const wordTargets = {
      'Short': '120-160 words',
      'Standard': '200-300 words',
      'Long': '350-500 words'
    };
    const lengthGuidance = wordTargets[length as keyof typeof wordTargets] || wordTargets.Standard;
    
    // Always default to Professional tone
    const tone = job.tone || 'Professional';
    const styleContext = `\n\nStyle: ${tone} tone, ${lengthGuidance}. Concise, investor-ready, no marketing fluff.`;

    return `${systemPrompt}${briefContext}${sectionContext}${taskContext}${userPromptContext}${editorContext}${styleContext}

Important:
- Write in markdown format
- Do not include a section title header - just the content
- Maximum 500 words
- No bracket placeholders
- If Business Brief data is missing, use neutral phrasing (e.g., "The company offers a range of services...")`;
  },

  rephrase: (job: AiJob) => {
    // Always default to Professional tone per SN's spec
    const tone = job.tone || "Professional";
    
    // Add Business Brief context for consistency
    const brief = job.context?.businessBrief;
    let briefContext = '';
    if (brief) {
      const briefParts = [];
      if (brief.company) briefParts.push(`Company: ${brief.company}`);
      if (brief.productsServices) briefParts.push(`Products/Services: ${brief.productsServices}`);
      if (brief.offeringsFull) briefParts.push(`Offerings: ${brief.offeringsFull}`);
      
      if (briefParts.length > 0) {
        briefContext = `\n\nBusiness Brief (for consistency):\n${briefParts.join('\n')}\n`;
      }
    }
    
    return `Rewrite the following business plan content for clarity. Keep the same meaning with shorter sentences and better flow.${briefContext}

Original content:
${job.existingContent}

Requirements:
- Rewrite with same meaning, shorter sentences, better flow
- Maintain all key facts, figures, and points
- Use ${tone.toLowerCase()} tone
- Use Business Brief context to keep names/offerings consistent
- Use markdown formatting
- Do not add new information`;
  },

  expand: (job: AiJob) => {
    // Always default to Professional tone per SN's spec
    const tone = job.tone || "Professional";
    
    // Add Business Brief context
    const brief = job.context?.businessBrief;
    let briefContext = '';
    if (brief) {
      const briefParts = [];
      if (brief.company) briefParts.push(`Company: ${brief.company}`);
      if (brief.productsServices) briefParts.push(`Products/Services: ${brief.productsServices}`);
      if (brief.offeringsFull) briefParts.push(`Full Suite: ${brief.offeringsFull}`);
      if (brief.market) briefParts.push(`Target Market: ${brief.market}`);
      if (brief.differentiator) briefParts.push(`Competitive Edge: ${brief.differentiator}`);
      
      if (briefParts.length > 0) {
        briefContext = `\n\nBusiness Brief:\n${briefParts.join('\n')}\n`;
      }
    }

    return `Add detail or examples to the following business plan content.${briefContext}

Current content:
${job.existingContent}

Requirements:
- Add specific details, examples, sub-bullets
- Stay within 30-40% longer max (avoid runaway length)
- Pull in Business Brief for specifics if missing
- Use markdown formatting
- Maintain ${tone.toLowerCase()} tone
- Maximum 500 words total`;
  },

  summarize: (job: AiJob) => {
    // Always default to Professional tone per SN's spec
    const tone = job.tone || "Professional";
    return `Make the following business plan content more concise.

Current content:
${job.existingContent}

Requirements:
- Condense to ~50-60% of original length
- Keep all critical points; remove fluff
- Keep all essential facts, figures, and main points
- Remove redundancy and unnecessary details
- Maintain logical flow and structure
- Maintain a ${tone.toLowerCase()} tone
- Use markdown formatting
- Aim to reduce the content by 30-50% while retaining value
- Do not lose any critical business information`;
  },
};

/**
 * Process an AI job for business plan section content
 * @param job - The AI job specification
 * @returns Promise containing the AI response
 */
export async function processAiSectionJob(job: AiJob): Promise<AiResponse> {
  try {
    // Validate required fields
    if (!job.action || !job.sectionKind || !job.sectionTitle) {
      return {
        success: false,
        error: "Missing required fields: action, sectionKind, and sectionTitle are required",
      };
    }

    // Default tone to Professional if not provided
    const tone = job.tone || "Professional";

    // Validate action type
    if (!['generate', 'rephrase', 'expand', 'summarize'].includes(job.action)) {
      return {
        success: false,
        error: `Invalid action: ${job.action}. Must be one of: generate, rephrase, expand, summarize`,
      };
    }

    // Validate tone if provided
    if (tone && !['Professional', 'Friendly', 'Bold', 'Minimal'].includes(tone)) {
      return {
        success: false,
        error: `Invalid tone: ${tone}. Must be one of: Professional, Friendly, Bold, Minimal`,
      };
    }

    // Check for existing content when required
    if (['rephrase', 'expand', 'summarize'].includes(job.action) && (!job.existingContent || job.existingContent.trim().length === 0)) {
      return {
        success: false,
        error: `Action "${job.action}" requires existing content to work with`,
      };
    }

    // Get system prompt for the selected tone
    const systemPrompt = TONE_SYSTEM_PROMPTS[job.tone as keyof typeof TONE_SYSTEM_PROMPTS];
    
    // Get action-specific prompt
    const actionPromptFunction = ACTION_PROMPTS[job.action as keyof typeof ACTION_PROMPTS];
    const userPrompt = actionPromptFunction(job);

    console.log(`🤖 Processing AI section job: ${job.action} for "${job.sectionTitle}" in ${job.tone} tone`);

    // First attempt with GPT-5 (only supports default temperature=1, no top_p)
    try {
      const startTime = Date.now();
      const completion = await openai.chat.completions.create({
        model: DEFAULT_CONFIG.model,
        messages: [
          {
            role: "system",
            content: systemPrompt,
          },
          {
            role: "user", 
            content: userPrompt,
          },
        ],
        max_completion_tokens: DEFAULT_CONFIG.maxTokens,
      });
      const latencyMs = Date.now() - startTime;

      const content = completion.choices[0]?.message?.content;
      if (!content) {
        throw new Error("No content received from GPT-5");
      }

      // Clean up the content (remove excessive whitespace, ensure proper formatting)
      const cleanedContent = content.trim().replace(/\n{3,}/g, '\n\n');

      // Log telemetry
      console.log(`✅ AI section job completed: ${job.action} for "${job.sectionTitle}" | model: ${DEFAULT_CONFIG.model} | latency: ${latencyMs}ms | prompt_tokens: ${completion.usage?.prompt_tokens || 0} | completion_tokens: ${completion.usage?.completion_tokens || 0} | fallback: false`);

      return {
        success: true,
        content: cleanedContent,
        usage: {
          tokens: completion.usage?.total_tokens || 0,
          model: DEFAULT_CONFIG.model,
        },
      };

    } catch (gpt5Error) {
      console.warn("GPT-5 failed, falling back to gpt-4o-mini:", gpt5Error);

      // Fallback to gpt-4o-mini (supports temperature and top_p)
      const fallbackStartTime = Date.now();
      const completion = await openai.chat.completions.create({
        model: "gpt-4o-mini",
        messages: [
          {
            role: "system",
            content: systemPrompt,
          },
          {
            role: "user",
            content: userPrompt,
          },
        ],
        temperature: DEFAULT_CONFIG.temperature,
        max_tokens: Math.min(DEFAULT_CONFIG.maxTokens, 4000),
        top_p: 0.9,
      });
      const fallbackLatencyMs = Date.now() - fallbackStartTime;

      const content = completion.choices[0]?.message?.content;
      if (!content) {
        throw new Error("No content received from gpt-4o-mini fallback");
      }

      const cleanedContent = content.trim().replace(/\n{3,}/g, '\n\n');

      // Log telemetry with fallback indicator
      console.log(`✅ AI section job completed: ${job.action} for "${job.sectionTitle}" | model: gpt-4o-mini | latency: ${fallbackLatencyMs}ms | prompt_tokens: ${completion.usage?.prompt_tokens || 0} | completion_tokens: ${completion.usage?.completion_tokens || 0} | fallback: true`);

      return {
        success: true,
        content: cleanedContent,
        usage: {
          tokens: completion.usage?.total_tokens || 0,
          model: "gpt-4o-mini",
        },
      };
    }

  } catch (error) {
    console.error("AI section job failed:", error);
    
    let errorMessage = "An unexpected error occurred while processing your request";
    
    if (error instanceof Error) {
      if (error.message.includes("insufficient_quota") || error.message.includes("billing")) {
        errorMessage = "AI service is temporarily unavailable due to quota limits. Please try again later.";
      } else if (error.message.includes("rate_limit")) {
        errorMessage = "Too many requests. Please wait a moment and try again.";
      } else if (error.message.includes("invalid_api_key") || error.message.includes("authentication")) {
        errorMessage = "AI service authentication error. Please contact support.";
      } else {
        errorMessage = `AI processing error: ${error.message}`;
      }
    }

    return {
      success: false,
      error: errorMessage,
    };
  }
}

// String coercion helper to guard against nulls/undefined
const S = (v?: string | null) => (v ?? '').trim();

// Generate structured template fields with AI
export async function generateStructuredTemplate(
  templateKey: string,
  businessBrief?: any,
  config: AiConfig = DEFAULT_CONFIG
): Promise<{ data: Record<string, string>, isSeedDraft?: boolean }> {
  // Seed drafts for each template type
  const seedDrafts: Record<string, Record<string, string>> = {
    swot: {
      strengths: '• Strong value proposition\n• Experienced team\n• Market opportunity',
      weaknesses: '• Limited brand awareness\n• Resource constraints\n• Market competition',
      opportunities: '• Growing market demand\n• Strategic partnerships\n• Product expansion',
      threats: '• Competitive pressure\n• Market shifts\n• Economic factors'
    },
    persona: {
      name: 'Target Customer',
      demographics: 'Age 25-45, urban, college-educated, mid-income',
      jobs: 'Seeking efficient solutions, looking to save time, wanting quality results',
      pain: 'Current solutions are too expensive, complex, or time-consuming',
      channels: 'Online search, social media, word of mouth, industry events'
    },
    porter5: {
      threat_new: 'Moderate barrier to entry; established players have brand advantage',
      bargain_sup: 'Multiple suppliers available; moderate switching costs',
      bargain_buy: 'Customers have alternatives; price sensitivity varies by segment',
      threat_sub: 'Some substitute products exist; differentiation is key',
      rivalry: 'Competitive market with several established players'
    },
    tam_sam_som: {
      tam: 'Total addressable market represents the overall industry opportunity',
      sam: 'Serviceable market focuses on our target segments and regions',
      som: 'Obtainable market reflects realistic capture rate in first 3 years',
      sources: 'Industry reports, market research, competitive analysis'
    },
    pricing_table: {
      rows: 'Starter – $29/mo – Essential features\nPro – $79/mo – Most popular\nEnterprise – Custom – Full suite'
    }
  };

  const templates: Record<string, { fields: string[], prompt: string }> = {
    swot: {
      fields: ['strengths', 'weaknesses', 'opportunities', 'threats'],
      prompt: `Using the Business Brief, generate concise SWOT bullets (S/W/O/T). Practical, skimmable. Return ONLY a JSON object with keys: strengths, weaknesses, opportunities, threats. Each value should be 2-4 concise bullet points starting with •. No placeholders.`
    },
    persona: {
      fields: ['name', 'demographics', 'jobs', 'pain', 'channels'],
      prompt: `1–2 primary personas: Name/Role, Goals, Pain Points, Buying Triggers. Short blocks. Return ONLY a JSON object with keys: name (creative persona name), demographics (age, location, income, education), jobs (goals/jobs-to-be-done), pain (pain points), channels (acquisition channels). Keep each field concise, no placeholders.`
    },
    porter5: {
      fields: ['threat_new', 'bargain_sup', 'bargain_buy', 'threat_sub', 'rivalry'],
      prompt: `Five Forces: 1–2 lines each tied to our market. Return ONLY a JSON object with keys: threat_new (threat of new entrants), bargain_sup (supplier power), bargain_buy (buyer power), threat_sub (substitutes), rivalry (industry rivalry). Each 1-2 sentences, no placeholders.`
    },
    tam_sam_som: {
      fields: ['tam', 'sam', 'som', 'sources'],
      prompt: `Rough, logical size estimates; short justification; simple bullets. Return ONLY a JSON object with keys: tam (total addressable market), sam (serviceable available market), som (serviceable obtainable market), sources (research sources). Keep qualitative if numbers unknown, no placeholders.`
    },
    pricing_table: {
      fields: ['rows'],
      prompt: `3 tiers (Starter/Pro/Business): price points + key features; compact table-like bullets. Return ONLY a JSON object with key: rows. Format rows as: "Tier – $Price/mo – Key features" separated by \\n. No placeholders.`
    }
  };

  const template = templates[templateKey];
  if (!template) {
    throw new Error(`Unknown template: ${templateKey}`);
  }

  // Build context from business brief with null guards
  let context = '';
  if (businessBrief) {
    const parts = [];
    if (S(businessBrief.company)) parts.push(`Company: ${S(businessBrief.company)}`);
    if (S(businessBrief.industry)) parts.push(`Industry: ${S(businessBrief.industry)}`);
    if (S(businessBrief.businessModel)) parts.push(`Business Model: ${S(businessBrief.businessModel)}`);
    if (S(businessBrief.offeringsFull)) parts.push(`Offerings: ${S(businessBrief.offeringsFull)}`);
    if (S(businessBrief.market)) parts.push(`Target Market: ${S(businessBrief.market)}`);
    if (S(businessBrief.differentiator)) parts.push(`Competitive Edge: ${S(businessBrief.differentiator)}`);
    if (S(businessBrief.financialHeadline)) parts.push(`Financial Headline: ${S(businessBrief.financialHeadline)}`);
    
    if (parts.length > 0) {
      context = `Business Context:\n${parts.join('\n')}\n\n`;
    }
  }

  const fullPrompt = `${context}${template.prompt}

IMPORTANT: Return ONLY valid JSON, no markdown, no explanation, no code blocks. Example format: {"field1": "value", "field2": "value"}`;

  // Primary attempt with GPT-5 (no temperature, only default=1)
  try {
    const startTime = Date.now();
    console.log(`🎯 Generating structured template: ${templateKey} (GPT-5 primary)`);
    
    const completion = await openai.chat.completions.create({
      model: "gpt-5",
      messages: [
        {
          role: "system",
          content: "You are a business analysis expert. Generate structured, concise business plan content. Always return valid JSON only."
        },
        {
          role: "user",
          content: fullPrompt
        }
      ],
      max_completion_tokens: 1500,
    });

    const latencyMs = Date.now() - startTime;
    const responseText = completion.choices[0]?.message?.content?.trim();
    const promptTokens = completion.usage?.prompt_tokens || 0;
    const completionTokens = completion.usage?.completion_tokens || 0;
    
    // Log telemetry
    console.log(`✅ Template generation: ${templateKey} | model: gpt-5 | latency: ${latencyMs}ms | prompt_tokens: ${promptTokens} | completion_tokens: ${completionTokens} | fallback: false`);
    
    if (!responseText || completionTokens === 0) {
      throw new Error("No content generated from GPT-5");
    }
    
    // Clean up response - remove markdown code blocks if present
    let jsonText = responseText;
    if (jsonText.startsWith('```')) {
      jsonText = jsonText.replace(/```json\n?/g, '').replace(/```\n?/g, '').trim();
    }
    
    const parsedData = JSON.parse(jsonText);
    
    // Validate fields exist and have content
    const result: Record<string, string> = {};
    let hasAnyContent = false;
    for (const field of template.fields) {
      const value = S(parsedData[field]);
      result[field] = value;
      if (value) hasAnyContent = true;
    }
    
    if (!hasAnyContent) {
      throw new Error("Parsed JSON has no content");
    }
    
    return { data: result, isSeedDraft: false };

  } catch (gpt5Error) {
    console.warn(`⚠️ GPT-5 failed for ${templateKey}, falling back to gpt-4o-mini:`, gpt5Error);

    // Fallback to gpt-4o-mini
    try {
      const fallbackStartTime = Date.now();
      const fallbackCompletion = await openai.chat.completions.create({
        model: "gpt-4o-mini",
        messages: [
          {
            role: "system",
            content: "You are a business analysis expert. Generate structured, concise business plan content. Always return valid JSON only."
          },
          {
            role: "user",
            content: fullPrompt
          }
        ],
        temperature: 0.7,
        max_tokens: 1500,
      });

      const fallbackLatencyMs = Date.now() - fallbackStartTime;
      const fallbackText = fallbackCompletion.choices[0]?.message?.content?.trim();
      const fallbackPromptTokens = fallbackCompletion.usage?.prompt_tokens || 0;
      const fallbackCompletionTokens = fallbackCompletion.usage?.completion_tokens || 0;
      
      // Log telemetry with fallback indicator
      console.log(`✅ Template generation: ${templateKey} | model: gpt-4o-mini | latency: ${fallbackLatencyMs}ms | prompt_tokens: ${fallbackPromptTokens} | completion_tokens: ${fallbackCompletionTokens} | fallback: true`);
      
      if (!fallbackText || fallbackCompletionTokens === 0) {
        throw new Error("No content from gpt-4o-mini fallback");
      }

      // Clean up response
      let jsonText = fallbackText;
      if (jsonText.startsWith('```')) {
        jsonText = jsonText.replace(/```json\n?/g, '').replace(/```\n?/g, '').trim();
      }
      
      const parsedData = JSON.parse(jsonText);
      
      // Validate fields
      const result: Record<string, string> = {};
      let hasAnyContent = false;
      for (const field of template.fields) {
        const value = S(parsedData[field]);
        result[field] = value;
        if (value) hasAnyContent = true;
      }
      
      if (!hasAnyContent) {
        throw new Error("Fallback JSON has no content");
      }
      
      return { data: result, isSeedDraft: false };

    } catch (fallbackError) {
      console.error(`❌ Both GPT-5 and fallback failed for ${templateKey}, using seed draft:`, fallbackError);
      
      // Return seed draft as last resort
      const seedDraft = seedDrafts[templateKey] || {};
      console.log(`🌱 Using seed draft for ${templateKey}`);
      return { data: seedDraft, isSeedDraft: true };
    }
  }
}

// Fallback content generator for when AI is completely unavailable
export function generateFallbackContent(job: AiJob): AiResponse {
  console.log("🔄 Generating fallback content for AI section job");
  
  let content = "";
  
  switch (job.action) {
    case "generate":
      content = `## ${job.sectionTitle}

This section of your business plan is ready for your input. Consider including:

- Key objectives and goals for this area
- Relevant details specific to your business
- Supporting data and evidence
- Strategic considerations and next steps

*This is placeholder content. Please replace with your specific business information.*`;
      break;
      
    case "rephrase":
      content = job.existingContent || "Content to be rephrased";
      break;
      
    case "expand":
      content = `${job.existingContent || ""}

*Additional details and expansion would be added here based on your specific business context and requirements.*`;
      break;
      
    case "summarize":
      content = job.existingContent?.slice(0, Math.floor(job.existingContent.length * 0.6)) || "Summary content";
      break;
  }

  return {
    success: true,
    content,
    usage: {
      tokens: 0,
      model: "fallback",
    },
  };
}