import OpenAI from "openai";

// the newest OpenAI model is "gpt-5" which was released August 7, 2025. do not change this unless explicitly requested by the user

export type PlanMode = "lean" | "full";
export type PlanTone = "Professional" | "Friendly" | "Bold" | "Minimal";

export interface PlanRequest {
  mode: PlanMode;
  businessName: string;
  description: string;
  industry?: string;
  audience?: string;
  problem?: string;
  solution?: string;
  advantage?: string;
  pricing?: string;
  timeframeMonths?: number;
  tone?: PlanTone;
}

export interface PlanSection {
  id: string;
  title: string;
  content: string;
}

export interface FinanceSnapshot {
  assumptions: {
    revenueGrowthMonthly: number;
    customerAcquisitionCost: number;
    averageRevenuePerUser: number;
    churnRateMonthly: number;
  };
  projections: Array<{
    month: number;
    revenue: number;
    expenses: number;
    profit: number;
    customers: number;
  }>;
  summary: {
    breakEvenMonth: number;
    totalRevenue12Month: number;
    totalProfit12Month: number;
  };
}

export interface PlanResponse {
  mode: PlanMode;
  sections: PlanSection[];
  finance?: FinanceSnapshot;
}

const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
});

// Tone-specific system prompts for business plan generation
const TONE_SYSTEM_PROMPTS = {
  Professional: `You are a senior business strategy consultant with extensive experience helping startups and established companies develop comprehensive business plans. Write in a polished, confident, and authoritative tone. Use business terminology appropriately and maintain a formal, credible style throughout.`,

  Friendly: `You are an approachable business mentor who explains complex business concepts in warm, encouraging language. Write in a conversational but professional tone, using inclusive language that makes the reader feel supported and motivated. Avoid jargon when simpler terms work better.`,

  Bold: `You are a dynamic business strategist who writes with confidence and conviction. Use strong, impactful language that inspires action. Be direct and decisive in your recommendations. Emphasize competitive advantages and ambitious goals with energetic, compelling prose.`,

  Minimal: `You are a strategic business advisor who values clarity and brevity. Write in clean, concise language that gets straight to the point. Use short sentences and bullet points when appropriate. Eliminate unnecessary words while maintaining substance and actionability.`
};

// Lean plan sections (5 sections for free users)
const LEAN_PLAN_SECTIONS = [
  { id: "executive_summary", title: "Executive Summary" },
  { id: "customer_problem", title: "Customer & Problem" },
  { id: "offer", title: "Your Offer" },
  { id: "go_to_market", title: "Go-to-Market Strategy" },
  { id: "objectives_90_day", title: "90-Day Objectives" }
];

// Full plan sections (10+ sections for pro users)
const FULL_PLAN_SECTIONS = [
  { id: "executive_summary", title: "Executive Summary" },
  { id: "company_overview", title: "Company Overview" },
  { id: "market_analysis", title: "Market Analysis" },
  { id: "customer_problem", title: "Customer & Problem" },
  { id: "solution_product", title: "Solution & Product" },
  { id: "competitive_advantage", title: "Competitive Advantage" },
  { id: "business_model", title: "Business Model & Revenue" },
  { id: "marketing_strategy", title: "Marketing & Sales Strategy" },
  { id: "operations", title: "Operations & Implementation" },
  { id: "risk_mitigation", title: "Risk Assessment & Mitigation" },
  { id: "milestones", title: "Key Milestones & Timeline" },
  { id: "next_steps", title: "Immediate Next Steps" }
];

function getLeanPlanPrompt(request: PlanRequest): string {
  const {
    businessName,
    description,
    industry = "Not specified",
    audience = "Not specified",
    problem = "Not specified",
    solution = "Not specified",
    advantage = "Not specified",
    pricing = "Not specified",
    timeframeMonths = 12
  } = request;

  return `Generate a lean business plan for: ${businessName}

Business Details:
- Description: ${description}
- Industry: ${industry}
- Target Audience: ${audience}
- Problem Solved: ${problem}
- Solution Approach: ${solution}
- Key Advantage: ${advantage}
- Pricing Strategy: ${pricing}
- Planning Timeframe: ${timeframeMonths} months

Create exactly 5 sections as specified below. Each section should be 50-110 words, actionable, and specific to this business.

Required sections:
1. Executive Summary - Brief overview of the business, its mission, and key value proposition
2. Customer & Problem - Who your customers are and what specific problem you solve for them
3. Your Offer - What you're selling and how it addresses the customer problem
4. Go-to-Market Strategy - How you'll reach customers and acquire them initially
5. 90-Day Objectives - Specific, measurable goals for the first 90 days

Return ONLY a valid JSON object in this exact format:
{
  "sections": [
    {"id": "executive_summary", "title": "Executive Summary", "content": "..."},
    {"id": "customer_problem", "title": "Customer & Problem", "content": "..."},
    {"id": "offer", "title": "Your Offer", "content": "..."},
    {"id": "go_to_market", "title": "Go-to-Market Strategy", "content": "..."},
    {"id": "objectives_90_day", "title": "90-Day Objectives", "content": "..."}
  ]
}`;
}

function getFullPlanPrompt(request: PlanRequest): string {
  const {
    businessName,
    description,
    industry = "Not specified",
    audience = "Not specified",
    problem = "Not specified",
    solution = "Not specified",
    advantage = "Not specified",
    pricing = "Not specified",
    timeframeMonths = 12
  } = request;

  return `Generate a comprehensive business plan for: ${businessName}

Business Details:
- Description: ${description}
- Industry: ${industry}
- Target Audience: ${audience}
- Problem Solved: ${problem}
- Solution Approach: ${solution}
- Key Advantage: ${advantage}
- Pricing Strategy: ${pricing}
- Planning Timeframe: ${timeframeMonths} months

Create exactly 12 sections as specified below. Each section should be 80-180 words, detailed, and actionable.

Also include a financial snapshot with realistic projections for ${timeframeMonths} months.

Return ONLY a valid JSON object in this exact format:
{
  "sections": [
    {"id": "executive_summary", "title": "Executive Summary", "content": "..."},
    {"id": "company_overview", "title": "Company Overview", "content": "..."},
    {"id": "market_analysis", "title": "Market Analysis", "content": "..."},
    {"id": "customer_problem", "title": "Customer & Problem", "content": "..."},
    {"id": "solution_product", "title": "Solution & Product", "content": "..."},
    {"id": "competitive_advantage", "title": "Competitive Advantage", "content": "..."},
    {"id": "business_model", "title": "Business Model & Revenue", "content": "..."},
    {"id": "marketing_strategy", "title": "Marketing & Sales Strategy", "content": "..."},
    {"id": "operations", "title": "Operations & Implementation", "content": "..."},
    {"id": "risk_mitigation", "title": "Risk Assessment & Mitigation", "content": "..."},
    {"id": "milestones", "title": "Key Milestones & Timeline", "content": "..."},
    {"id": "next_steps", "title": "Immediate Next Steps", "content": "..."}
  ],
  "finance": {
    "assumptions": {
      "revenueGrowthMonthly": 15,
      "customerAcquisitionCost": 50,
      "averageRevenuePerUser": 100,
      "churnRateMonthly": 5
    },
    "projections": [
      {"month": 1, "revenue": 1000, "expenses": 2000, "profit": -1000, "customers": 10},
      {"month": 2, "revenue": 1500, "expenses": 2100, "profit": -600, "customers": 15},
      {"month": 3, "revenue": 2250, "expenses": 2200, "profit": 50, "customers": 23}
    ],
    "summary": {
      "breakEvenMonth": 3,
      "totalRevenue12Month": 50000,
      "totalProfit12Month": 15000
    }
  }
}

Provide realistic financial projections based on the business model. Include monthly data for the full timeframe. Ensure all numbers are realistic for the described business.`;
}

function parseBusinessPlanResponse(content: string, mode: PlanMode): PlanResponse {
  try {
    const parsed = JSON.parse(content);
    
    if (!parsed.sections || !Array.isArray(parsed.sections)) {
      throw new Error("Invalid response format: missing or invalid sections array");
    }

    // Validate sections structure
    const expectedSections = mode === "lean" ? LEAN_PLAN_SECTIONS : FULL_PLAN_SECTIONS;
    const sections: PlanSection[] = [];

    for (const expectedSection of expectedSections) {
      const section = parsed.sections.find((s: any) => s.id === expectedSection.id);
      if (!section || !section.content) {
        throw new Error(`Missing required section: ${expectedSection.id}`);
      }

      // Apply word count caps
      const maxWords = mode === "lean" ? 110 : 180;
      const words = section.content.split(/\s+/);
      const cappedContent = words.length > maxWords 
        ? words.slice(0, maxWords).join(' ') + '...'
        : section.content;

      sections.push({
        id: section.id,
        title: section.title || expectedSection.title,
        content: cappedContent
      });
    }

    const response: PlanResponse = {
      mode,
      sections
    };

    // Add finance data for full plans
    if (mode === "full" && parsed.finance) {
      response.finance = parsed.finance;
    }

    return response;
  } catch (error) {
    console.error("Failed to parse business plan response:", error);
    throw new Error("Invalid JSON response from AI model");
  }
}

function generateFallbackPlan(request: PlanRequest): PlanResponse {
  const { businessName, mode } = request;
  
  if (mode === "lean") {
    return {
      mode: "lean",
      sections: [
        {
          id: "executive_summary",
          title: "Executive Summary",
          content: `${businessName} is an innovative business focused on delivering value to customers through our unique approach. We aim to solve key market problems while building a sustainable and profitable operation.`
        },
        {
          id: "customer_problem",
          title: "Customer & Problem",
          content: `Our target customers face significant challenges in their daily operations. ${businessName} addresses these pain points by providing efficient, cost-effective solutions that save time and improve outcomes.`
        },
        {
          id: "offer",
          title: "Your Offer",
          content: `We offer a comprehensive solution that combines quality, affordability, and excellent customer service. Our approach is designed to meet customer needs while maintaining competitive pricing.`
        },
        {
          id: "go_to_market",
          title: "Go-to-Market Strategy",
          content: `Our initial market entry will focus on digital marketing, word-of-mouth referrals, and strategic partnerships. We'll prioritize customer satisfaction to build a strong reputation and organic growth.`
        },
        {
          id: "objectives_90_day",
          title: "90-Day Objectives",
          content: `Within 90 days, we aim to establish our initial customer base, refine our offering based on feedback, and build operational processes that support sustainable growth.`
        }
      ]
    };
  } else {
    return {
      mode: "full",
      sections: [
        {
          id: "executive_summary",
          title: "Executive Summary",
          content: `${businessName} represents a strategic opportunity to capture market share in our target industry. With a clear value proposition and strong execution plan, we are positioned for sustainable growth and profitability.`
        },
        {
          id: "company_overview",
          title: "Company Overview",
          content: `${businessName} is structured to deliver exceptional value while maintaining operational efficiency. Our team brings together diverse expertise to execute our business strategy effectively.`
        },
        {
          id: "market_analysis",
          title: "Market Analysis",
          content: `The market presents significant opportunities for growth, with increasing demand for solutions like ours. We've identified key trends that support our business model and growth projections.`
        },
        {
          id: "customer_problem",
          title: "Customer & Problem",
          content: `Our research indicates that customers in our target market face persistent challenges that current solutions fail to address adequately. This creates a clear opportunity for ${businessName}.`
        },
        {
          id: "solution_product",
          title: "Solution & Product",
          content: `Our solution is designed to directly address customer pain points through innovative features and superior user experience. We focus on delivering measurable value and positive outcomes.`
        },
        {
          id: "competitive_advantage",
          title: "Competitive Advantage",
          content: `${businessName} differentiates itself through unique capabilities, superior customer service, and strategic positioning. Our competitive moat will strengthen over time through network effects.`
        },
        {
          id: "business_model",
          title: "Business Model & Revenue",
          content: `Our revenue model is designed for predictability and scalability. We've structured pricing to align customer value with business sustainability, creating win-win relationships.`
        },
        {
          id: "marketing_strategy",
          title: "Marketing & Sales Strategy",
          content: `Our marketing approach combines digital channels, content marketing, and strategic partnerships. We focus on building trust and demonstrating value through education and social proof.`
        },
        {
          id: "operations",
          title: "Operations & Implementation",
          content: `Operational excellence is critical to our success. We've designed systems and processes that scale efficiently while maintaining quality standards and customer satisfaction.`
        },
        {
          id: "risk_mitigation",
          title: "Risk Assessment & Mitigation",
          content: `We've identified key business risks and developed mitigation strategies. Our approach includes diversification, contingency planning, and continuous monitoring of external factors.`
        },
        {
          id: "milestones",
          title: "Key Milestones & Timeline",
          content: `Our roadmap includes specific milestones for product development, customer acquisition, and revenue growth. Each milestone has defined success metrics and timeline targets.`
        },
        {
          id: "next_steps",
          title: "Immediate Next Steps",
          content: `The immediate priority is executing our launch strategy, establishing initial customer relationships, and iterating based on market feedback. Success requires disciplined execution.`
        }
      ],
      finance: {
        assumptions: {
          revenueGrowthMonthly: 10,
          customerAcquisitionCost: 75,
          averageRevenuePerUser: 150,
          churnRateMonthly: 3
        },
        projections: [
          { month: 1, revenue: 1500, expenses: 3000, profit: -1500, customers: 10 },
          { month: 2, revenue: 2250, expenses: 3200, profit: -950, customers: 15 },
          { month: 3, revenue: 3375, expenses: 3400, profit: -25, customers: 23 },
          { month: 4, revenue: 5062, expenses: 3600, profit: 1462, customers: 34 },
          { month: 5, revenue: 7593, expenses: 3800, profit: 3793, customers: 51 },
          { month: 6, revenue: 11390, expenses: 4000, profit: 7390, customers: 76 }
        ],
        summary: {
          breakEvenMonth: 4,
          totalRevenue12Month: 150000,
          totalProfit12Month: 45000
        }
      }
    };
  }
}

export async function generateBusinessPlan(request: PlanRequest): Promise<PlanResponse> {
  const { 
    mode = "lean",
    businessName,
    description,
    tone = "Professional"
  } = request;

  if (!businessName || !description) {
    throw new Error("businessName and description are required");
  }

  // Get appropriate system message for tone
  const systemMessage = TONE_SYSTEM_PROMPTS[tone];

  // Generate the appropriate prompt based on mode
  const prompt = mode === "lean" 
    ? getLeanPlanPrompt(request)
    : getFullPlanPrompt(request);

  try {
    // First attempt with GPT-5
    const completion = await openai.chat.completions.create({
      model: "gpt-5",
      messages: [
        {
          role: "system",
          content: systemMessage
        },
        {
          role: "user",
          content: prompt
        }
      ],
      temperature: 0.7,
      top_p: 0.9,
      max_completion_tokens: mode === "lean" ? 1200 : 3000
    });

    const content = completion.choices[0]?.message?.content;
    if (!content) {
      throw new Error("No response from OpenAI");
    }

    // Parse and validate the response
    return parseBusinessPlanResponse(content, mode);

  } catch (error) {
    console.log("GPT-5 failed, falling back to gpt-4o-mini:", error);
    
    try {
      // Fallback to gpt-4o-mini
      const completion = await openai.chat.completions.create({
        model: "gpt-4o-mini",
        messages: [
          {
            role: "system",
            content: systemMessage
          },
          {
            role: "user",
            content: prompt
          }
        ],
        temperature: 0.7,
        top_p: 0.9,
        max_completion_tokens: mode === "lean" ? 1200 : 3000
      });

      const content = completion.choices[0]?.message?.content;
      if (!content) {
        throw new Error("No response from OpenAI fallback");
      }

      // Parse and validate the response
      return parseBusinessPlanResponse(content, mode);

    } catch (fallbackError) {
      console.error("Both GPT-5 and gpt-4o-mini failed:", fallbackError);
      
      // Return fallback plan if all AI attempts fail
      console.log("Using fallback business plan");
      return generateFallbackPlan(request);
    }
  }
}