import OpenAI from "openai";

export interface GenerateBusinessNamesRequest {
  industry: string;
  keywords: string;
  style?: string;
  count?: number;
  creativity?: number; // 0-100, maps to temperature 0.2-1.0
  description?: string;
  vibe?: string; // single vibe selection
  length_range?: [number, number];
  starts_with?: string;
  must_include?: string; // comma-separated list
  must_exclude?: string; // comma-separated list
}

export interface GeneratedBusinessNamesResponse {
  names: string[];
}

// Simple blocklists for filtering
const TRADEMARK_BLOCKLIST = [
  'apple', 'google', 'microsoft', 'amazon', 'facebook', 'meta', 'tesla', 'netflix',
  'uber', 'airbnb', 'paypal', 'visa', 'mastercard', 'walmart', 'target', 'starbucks'
];

const NSFW_BLOCKLIST = [
  'sex', 'porn', 'nude', 'adult', 'escort', 'casino', 'gambling', 'drug', 'weed',
  'marijuana', 'cocaine', 'heroin', 'meth', 'weapon', 'gun', 'bomb', 'terror'
];

// Helper functions for post-processing
function normalizeString(str: string): string {
  return str.trim().toLowerCase().replace(/\s+/g, '');
}

function isValidBusinessName(name: string): boolean {
  // Allow letters, spaces, and hyphens for 1-2 word names
  return /^[a-zA-Z][a-zA-Z\s-]*[a-zA-Z]$/.test(name) && name.trim().split(/\s+/).length <= 2;
}

function filterBlocklists(names: string[]): string[] {
  return names.filter(name => {
    const normalized = normalizeString(name);
    
    // Check trademark blocklist
    if (TRADEMARK_BLOCKLIST.some(blocked => normalized.includes(blocked))) {
      return false;
    }
    
    // Check NSFW blocklist
    if (NSFW_BLOCKLIST.some(blocked => normalized.includes(blocked))) {
      return false;
    }
    
    return true;
  });
}

function deduplicateNames(names: string[]): string[] {
  const seen = new Set<string>();
  return names.filter(name => {
    const normalized = normalizeString(name);
    if (seen.has(normalized)) {
      return false;
    }
    seen.add(normalized);
    return true;
  });
}

const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
});

export async function generateBusinessNames(request: GenerateBusinessNamesRequest): Promise<GeneratedBusinessNamesResponse> {
  const { 
    industry, 
    keywords, 
    style = "Modern", 
    count = 12,
    creativity = 60,
    starts_with,
    must_include,
    must_exclude,
    length_range = [6, 10], // Default length range
    description,
    vibe
  } = request;

  // Map creativity (0-100) to temperature (0.2-1.0)
  const temperature = 0.2 + (creativity / 100) * 0.8;

  // Generate more names than requested to account for filtering
  const generateCount = Math.min(count * 2, 30);
  
  // Build prompt based on available inputs
  let businessContext = "";
  if (industry && keywords) {
    businessContext = `Generate ${generateCount} creative business names for a ${industry} business.`;
  } else if (industry) {
    businessContext = `Generate ${generateCount} creative business names for a ${industry} business.`;
  } else {
    businessContext = `Generate ${generateCount} creative business names inspired by these concepts.`;
  }

  let prompt = businessContext + `\n\nStyle: ${vibe || style || "Modern"}`;
  
  if (keywords) {
    prompt += `\nKeywords to incorporate or inspire: ${keywords}`;
  }
  if (industry) {
    prompt += `\nIndustry/Category: ${industry}`;
  }

  if (description) {
    prompt += `\nBusiness description: ${description}`;
  }

  // Add filtering requirements to the prompt
  prompt += `\n\nSpecial Requirements:`;
  if (starts_with) {
    prompt += `\n- ALL names MUST start with "${starts_with}"`;
  }
  if (must_include) {
    prompt += `\n- Names MUST include one of these words: ${must_include}`;
  }
  if (must_exclude) {
    prompt += `\n- Names MUST NOT contain any of these words: ${must_exclude}`;
  }
  if (length_range) {
    const [min, max] = length_range;
    prompt += `\n- Names must be between ${min} and ${max} characters long`;
  }

  prompt += `\n\nGeneral Requirements:
- Focus on short, pronounceable patterns (CVVC/VCCV/CVCV)
- Keep names ≤10 characters unless specified otherwise
- Avoid numbers, symbols, and clichéd terms
- Names should be memorable, brandable, and professional
- Mix of approaches: descriptive, invented, compound words
- Avoid trademark-like terms and existing brand names
- Each name should be 1-2 words maximum

Return ONLY a JSON array of strings (business names only):
["BusinessName1", "BusinessName2", "BusinessName3", ...]

Focus on creating names that:
- Sound professional and trustworthy
- Are easy to pronounce and remember${industry ? `
- Work well for ${industry} industry` : ''}
- Reflect ${vibe || style || "Modern"} style characteristics
- Can be inspired by: ${keywords}
- Are short and brandable (avoid long descriptive phrases)`;

  try {
    const completion = await openai.chat.completions.create({
      model: "gpt-4o-mini",
      messages: [
        {
          role: "system",
          content: "You are a professional brand naming expert. Generate creative, memorable business names that are brandable and professional. Always return valid JSON array format with only the business names as strings."
        },
        {
          role: "user",
          content: prompt
        }
      ],
      temperature: temperature,
      top_p: 0.9,
      max_tokens: 1000,
    });

    const content = completion.choices[0]?.message?.content;
    if (!content) {
      throw new Error("No response from OpenAI");
    }

    // Parse the JSON response - expect array of strings
    let rawNames: string[];
    try {
      // Remove markdown code block wrapping if present
      let cleanContent = content.trim();
      if (cleanContent.startsWith('```json')) {
        cleanContent = cleanContent.replace(/^```json\s*/, '').replace(/\s*```$/, '');
      }
      if (cleanContent.startsWith('```')) {
        cleanContent = cleanContent.replace(/^```\s*/, '').replace(/\s*```$/, '');
      }
      
      rawNames = JSON.parse(cleanContent);
      
      // Ensure we have an array of strings
      if (!Array.isArray(rawNames)) {
        throw new Error('Expected array of strings');
      }
      
      // Convert any non-strings to strings and filter out invalid entries
      rawNames = rawNames
        .map(item => typeof item === 'string' ? item : String(item))
        .filter(name => name && name.trim().length > 0)
        .map(name => name.trim());
        
    } catch (parseError) {
      console.error("Failed to parse OpenAI response:", content);
      // Fallback to default names if parsing fails
      rawNames = [
        "BrightPath", "NextGen", "Velocity", "Innovation", "DigitalForge", 
        "Momentum", "CoreTech", "PrimeVenture", "SwiftFlow", "NovaLabs",
        "Zenith", "Catalyst", "Fusion", "Apex", "Vertex", "Matrix"
      ];
    }

    // Post-process the names
    let processedNames = rawNames
      // Remove invalid names (must be letters + optional single hyphen)
      .filter(isValidBusinessName)
      // Apply blocklist filtering
      .slice(); // Create copy for blocklist filtering
    
    processedNames = filterBlocklists(processedNames);
    
    // Apply user constraints
    if (starts_with) {
      const startsWithLower = starts_with.toLowerCase().trim();
      processedNames = processedNames.filter(name => 
        name.toLowerCase().startsWith(startsWithLower)
      );
    }
    
    if (must_include) {
      const includeWords = must_include.split(',').map(w => w.trim().toLowerCase()).filter(Boolean);
      processedNames = processedNames.filter(name => {
        const nameLower = name.toLowerCase();
        return includeWords.some(word => nameLower.includes(word));
      });
    }
    
    if (must_exclude) {
      const excludeWords = must_exclude.split(',').map(w => w.trim().toLowerCase()).filter(Boolean);
      processedNames = processedNames.filter(name => {
        const nameLower = name.toLowerCase();
        return !excludeWords.some(word => nameLower.includes(word));
      });
    }
    
    // Apply length range
    const [minLen, maxLen] = length_range;
    processedNames = processedNames.filter(name => 
      name.length >= minLen && name.length <= maxLen
    );
    
    // Deduplicate
    processedNames = deduplicateNames(processedNames);
    
    // Limit to requested count (max 20)
    const finalCount = Math.min(count, 20);
    processedNames = processedNames.slice(0, finalCount);
    
    // If we don't have enough names after filtering, add some fallbacks
    if (processedNames.length < finalCount) {
      const fallbacks = [
        "BrightCore", "NextFlow", "VelocityLab", "InnovateTech", "DigitalMint", 
        "MomentumHub", "CoreStream", "PrimeForge", "SwiftNova", "ZenithLab",
        "CatalystFlow", "FusionCore", "ApexMint", "VertexHub", "MatrixLab"
      ].filter(name => !processedNames.includes(name));
      
      // Apply same filtering to fallbacks
      let filteredFallbacks = fallbacks.filter(isValidBusinessName);
      filteredFallbacks = filterBlocklists(filteredFallbacks);
      
      if (starts_with) {
        const startsWithLower = starts_with.toLowerCase().trim();
        filteredFallbacks = filteredFallbacks.filter(name => 
          name.toLowerCase().startsWith(startsWithLower)
        );
      }
      
      if (must_include) {
        const includeWords = must_include.split(',').map(w => w.trim().toLowerCase()).filter(Boolean);
        filteredFallbacks = filteredFallbacks.filter(name => {
          const nameLower = name.toLowerCase();
          return includeWords.some(word => nameLower.includes(word));
        });
      }
      
      if (must_exclude) {
        const excludeWords = must_exclude.split(',').map(w => w.trim().toLowerCase()).filter(Boolean);
        filteredFallbacks = filteredFallbacks.filter(name => {
          const nameLower = name.toLowerCase();
          return !excludeWords.some(word => nameLower.includes(word));
        });
      }
      
      filteredFallbacks = filteredFallbacks.filter(name => 
        name.length >= minLen && name.length <= maxLen
      );
      
      // Add fallbacks up to the requested count
      const needed = finalCount - processedNames.length;
      processedNames.push(...filteredFallbacks.slice(0, needed));
    }

    return { names: processedNames };

  } catch (error) {
    console.error("Error generating business names:", error);
    
    // Return fallback names if AI generation fails
    return { 
      names: [
        "BrightPath", "NextGen", "Velocity", "Innovation", "DigitalForge", 
        "Momentum", "CoreTech", "PrimeVenture", "SwiftFlow", "NovaLabs"
      ].slice(0, Math.min(count, 10)) 
    };
  }
}