import { Router } from "express";
import { verifyAdminBearer, setUserRole, getUserRole, hasRolePermission, isAdminRole } from "./firebaseAdmin";
import { logAuditEvent, getRecentAuditEvents, getAuditEventsForUser } from "./auditLogger";
import { storage } from "../storage";
import { 
  requireOwner, 
  requireManagementOrAbove, 
  requireStaffOrAbove, 
  requireAnalystOrAbove,
  devBypass,
  type AuthenticatedRequest 
} from "../middleware/authz";
import type { Role } from "../../shared/schema";
import { CANCELLATION_REASONS } from "../../shared/schema";
import Stripe from "stripe";

const router = Router();

// Lazy Stripe initialization - only when key is available
let stripe: Stripe | null = null;
function getStripe(): Stripe | null {
  if (stripe) return stripe;
  
  const stripeKey = process.env.STRIPE_SECRET_KEY;
  if (!stripeKey) {
    console.warn('STRIPE_SECRET_KEY not found - Stripe functionality will be disabled');
    return null;
  }
  
  try {
    stripe = new Stripe(stripeKey, {
      apiVersion: "2023-10-16",
    });
    return stripe;
  } catch (error) {
    console.error('Failed to initialize Stripe:', error);
    return null;
  }
}

// Date helpers for metrics
const startOfMonth = () => new Date(new Date().getFullYear(), new Date().getMonth(), 1);
const startOfYear = () => new Date(new Date().getFullYear(), 0, 1);

router.get("/api/admin/health", async (req, res) => {
  try {
    // In development, bypass authentication check for health endpoint
    if (process.env.NODE_ENV === 'development' || !process.env.FIREBASE_PROJECT_ID) {
      return res.json({ 
        ok: true, 
        ts: Date.now(),
        mode: 'development',
        firebase: 'bypassed',
        database: 'http-connection',
        roleSystem: 'active'
      });
    }
    
    const adminUser = await verifyAdminBearer(req.headers.authorization, 'staff');
    res.json({ 
      ok: true, 
      ts: Date.now(), 
      mode: 'production',
      userRole: adminUser.role,
      roleSystem: 'active'
    });
  } catch (error) {
    res.status(403).json({ 
      error: "Forbidden",
      details: process.env.NODE_ENV === 'development' ? 'Firebase Admin SDK not configured' : 'Authentication failed'
    });
  }
});

// Example protected op - get user count (staff level access)
router.get("/api/admin/users/count", devBypass('staff'), requireStaffOrAbove, async (req: AuthenticatedRequest, res) => {
  try {
    logAuditEvent({
      adminUserId: req.user!.uid,
      adminRole: req.user!.role,
      action: 'VIEW_USER_COUNT',
      details: 'Requested user count statistics'
    });
    
    // Get user count from storage
    const users = await storage.getAllUsers();
    res.json({ users: users.length });
  } catch (error) {
    console.error("Admin users/count error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
});

// Get all users with details (management level access required)
router.get("/api/admin/users", devBypass('management'), requireManagementOrAbove, async (req: AuthenticatedRequest, res) => {
  try {
    logAuditEvent({
      adminUserId: req.user!.uid,
      adminRole: req.user!.role,
      action: 'VIEW_ALL_USERS',
      details: 'Accessed full user database'
    });
    
    const users = await storage.getAllUsers();
    res.json({ users });
  } catch (error) {
    console.error("Admin users error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
});

// Get platform stats (analyst level access required)
router.get("/api/admin/stats", devBypass('analyst'), requireAnalystOrAbove, async (req: AuthenticatedRequest, res) => {
  try {
    logAuditEvent({
      adminUserId: req.user!.uid,
      adminRole: req.user!.role,
      action: 'VIEW_PLATFORM_STATS',
      details: 'Accessed platform statistics'
    });

    // In development, get real data directly from database using SQL
    if (process.env.NODE_ENV === 'development') {
      try {
        // Use HTTP-based connection instead of WebSocket to bypass connection issues
        const { neon } = await import("@neondatabase/serverless");
        const sql = neon(process.env.DATABASE_URL!);
        
        // Get counts using HTTP connection with Free/Pro breakdown
        const userCountResult = await sql`SELECT COUNT(*) as count FROM users`;
        const freeUsersResult = await sql`SELECT COUNT(*) as count FROM users WHERE is_paid = false OR is_paid IS NULL`;
        const proUsersResult = await sql`SELECT COUNT(*) as count FROM users WHERE is_paid = true`;
        
        // Get new paying users by time periods
        const today = new Date();
        const startOfDay = new Date(today.getFullYear(), today.getMonth(), today.getDate());
        const startOfWeek = new Date(today.setDate(today.getDate() - today.getDay()));
        const startOfMonth = new Date(today.getFullYear(), today.getMonth(), 1);
        
        const newPayingTodayResult = await sql`SELECT COUNT(*) as count FROM users WHERE is_paid = true AND pro_activated_at >= ${startOfDay.toISOString()}`;
        const newPayingWeekResult = await sql`SELECT COUNT(*) as count FROM users WHERE is_paid = true AND pro_activated_at >= ${startOfWeek.toISOString()}`;
        const newPayingMonthResult = await sql`SELECT COUNT(*) as count FROM users WHERE is_paid = true AND pro_activated_at >= ${startOfMonth.toISOString()}`;
        
        // Get recent users
        const recentUsersResult = await sql`SELECT id, email, display_name, created_at, is_paid FROM users ORDER BY created_at DESC LIMIT 5`;
        
        const stats = {
          totalUsers: parseInt(userCountResult[0].count),
          freeUsers: parseInt(freeUsersResult[0].count),
          proUsers: parseInt(proUsersResult[0].count),
          newPayingUsersToday: parseInt(newPayingTodayResult[0].count),
          newPayingUsersThisWeek: parseInt(newPayingWeekResult[0].count),
          newPayingUsersThisMonth: parseInt(newPayingMonthResult[0].count),
          recentUsers: recentUsersResult.map(u => ({
            id: u.id,
            email: u.email,
            displayName: u.display_name,
            createdAt: u.created_at,
            isPaid: u.is_paid
          }))
        };
        
        console.log("✅ Successfully fetched real database stats:", stats);
        return res.json(stats);
      } catch (sqlError) {
        console.log("Direct SQL failed, trying storage layer:", sqlError);
        
        // Fallback to storage layer
        try {
          const users = await storage.getAllUsers();
          const brandKits = await storage.getAllBrandKits();
          const businessNames = await storage.getAllBusinessNames();
          
          // Calculate Free vs Pro users
          const freeUsers = users.filter(u => !u.isPaid).length;
          const proUsers = users.filter(u => u.isPaid).length;
          
          const stats = {
            totalUsers: users.length,
            freeUsers,
            proUsers,
            totalBrandKits: brandKits.length,
            totalBusinessNames: businessNames.length,
            recentUsers: users
              .sort((a, b) => new Date(b.createdAt!).getTime() - new Date(a.createdAt!).getTime())
              .slice(0, 5)
              .map(u => ({
                id: u.id,
                email: u.email,
                displayName: u.displayName,
                createdAt: u.createdAt,
                isPaid: u.isPaid
              }))
          };
          
          return res.json(stats);
        } catch (storageError) {
          // Final fallback to mock data
          console.log("All database access failed, using mock data");
          return res.json({
            totalUsers: 42,
            freeUsers: 35,
            proUsers: 7,
            totalBrandKits: 18,
            totalBusinessNames: 156,
            recentUsers: [
              {
                id: '1',
                email: 'user1@example.com',
                displayName: 'Test User 1',
                createdAt: new Date().toISOString(),
                isPaid: false
              }
            ],
            note: "Using mock data due to database connectivity issues"
          });
        }
      }
    }

    // In production, use storage layer with Free/Pro breakdown
    const users = await storage.getAllUsers();
    const brandKits = await storage.getAllBrandKits();
    const businessNames = await storage.getAllBusinessNames();
    
    // Calculate Free vs Pro users
    const freeUsers = users.filter(u => !u.isPaid).length;
    const proUsers = users.filter(u => u.isPaid).length;
    
    const stats = {
      totalUsers: users.length,
      freeUsers,
      proUsers,
      totalBrandKits: brandKits.length,
      totalBusinessNames: businessNames.length,
      recentUsers: users
        .sort((a, b) => new Date(b.createdAt!).getTime() - new Date(a.createdAt!).getTime())
        .slice(0, 5)
        .map(u => ({
          id: u.id,
          email: u.email,
          displayName: u.displayName,
          createdAt: u.createdAt,
          isPaid: u.isPaid
        }))
    };
    
    res.json(stats);
  } catch (error) {
    console.error("Admin stats error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
});

// Get visitor analytics stats - daily visitor counts (analyst level access required)
router.get("/api/admin/visitor-stats", devBypass('analyst'), requireAnalystOrAbove, async (req: AuthenticatedRequest, res) => {
  try {
    logAuditEvent({
      adminUserId: req.user!.uid,
      adminRole: req.user!.role,
      action: 'VIEW_VISITOR_STATS',
      details: 'Accessed visitor analytics statistics'
    });

    // Get daily visitor stats for the last 30 days
    const endDate = new Date();
    const startDate = new Date();
    startDate.setDate(endDate.getDate() - 30);

    const startDateStr = startDate.toISOString().split('T')[0]; // YYYY-MM-DD
    const endDateStr = endDate.toISOString().split('T')[0];

    const dailyStats = await storage.getDailyVisitorStatsRange(startDateStr, endDateStr);

    // Fill in missing dates with zero values
    const statsMap = new Map();
    dailyStats.forEach(stat => {
      statsMap.set(stat.date, stat);
    });

    const filledStats = [];
    for (let d = new Date(startDate); d <= endDate; d.setDate(d.getDate() + 1)) {
      const dateStr = d.toISOString().split('T')[0];
      const stat = statsMap.get(dateStr);
      filledStats.push({
        date: dateStr,
        uniqueVisitors: stat?.uniqueVisitors || 0,
        totalPageViews: stat?.totalPageViews || 0,
        countries: stat?.countries || {},
        topPages: stat?.topPages || {}
      });
    }

    // Calculate totals for today and this week
    const today = endDateStr;
    const weekAgo = new Date();
    weekAgo.setDate(weekAgo.getDate() - 7);
    const weekAgoStr = weekAgo.toISOString().split('T')[0];

    const todayStats = statsMap.get(today);
    const weekStats = dailyStats.filter(stat => stat.date >= weekAgoStr);

    const totalToday = todayStats?.uniqueVisitors || 0;
    const totalThisWeek = weekStats.reduce((sum, stat) => sum + stat.uniqueVisitors, 0);

    res.json({
      dailyStats: filledStats,
      totalToday,
      totalThisWeek,
      dateRange: {
        start: startDateStr,
        end: endDateStr
      }
    });

  } catch (error) {
    console.error("Admin visitor stats error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
});

// Get visitor location data for world map (analyst level access required)
router.get("/api/admin/visitor-locations", devBypass('analyst'), requireAnalystOrAbove, async (req: AuthenticatedRequest, res) => {
  try {
    logAuditEvent({
      adminUserId: req.user!.uid,
      adminRole: req.user!.role,
      action: 'VIEW_VISITOR_LOCATIONS',
      details: 'Accessed visitor location data for world map'
    });

    // Get recent visitor sessions with location data (last 30 days)
    const recentSessions = await storage.getRecentVisitorSessions(30, 1000);

    // Group sessions by location for clustering and efficiency
    const locationMap = new Map();

    recentSessions.forEach(session => {
      if (session.latitude && session.longitude && session.country) {
        const key = `${session.latitude},${session.longitude}`;
        
        if (locationMap.has(key)) {
          const existing = locationMap.get(key);
          existing.count += session.pageViews;
          existing.sessions += 1;
        } else {
          locationMap.set(key, {
            latitude: parseFloat(session.latitude),
            longitude: parseFloat(session.longitude),
            country: session.country,
            countryCode: session.countryCode,
            city: session.city,
            count: session.pageViews,
            sessions: 1
          });
        }
      }
    });

    // Convert map to array for frontend consumption
    const locations = Array.from(locationMap.values())
      .filter(loc => loc.latitude && loc.longitude) // Ensure valid coordinates
      .sort((a, b) => b.count - a.count) // Sort by visitor count
      .slice(0, 500); // Limit to top 500 locations to avoid overwhelming the map

    // Also provide country-level statistics
    const countryStats = new Map();
    recentSessions.forEach(session => {
      if (session.countryCode && session.country) {
        const existing = countryStats.get(session.countryCode) || {
          country: session.country,
          countryCode: session.countryCode,
          visitors: 0,
          pageViews: 0
        };
        
        existing.visitors += 1;
        existing.pageViews += session.pageViews;
        countryStats.set(session.countryCode, existing);
      }
    });

    const countries = Array.from(countryStats.values())
      .sort((a, b) => b.visitors - a.visitors)
      .slice(0, 50); // Top 50 countries

    res.json({
      locations,
      countries,
      summary: {
        totalLocations: locations.length,
        totalCountries: countries.length,
        totalSessions: recentSessions.length,
        dateRange: {
          days: 30,
          description: 'Last 30 days'
        }
      }
    });

  } catch (error) {
    console.error("Admin visitor locations error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
});

// Role management endpoints (owner-only access)
router.post("/api/admin/users/:userId/role", devBypass('owner'), requireOwner, async (req: AuthenticatedRequest, res) => {
  try {
    const { userId } = req.params;
    const { role } = req.body;
    
    if (!role || !['owner', 'management', 'staff', 'analyst', 'user', 'pro'].includes(role)) {
      return res.status(400).json({ error: "Invalid role" });
    }
    
    // Look up the user to get their Firebase UID
    const user = await storage.getUser(userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    
    const firebaseUid = user.firebaseUid;
    
    // Prevent self-demotion from owner role
    if (req.user!.uid === firebaseUid && req.user!.role === 'owner' && role !== 'owner') {
      return res.status(400).json({ error: "Cannot demote yourself from owner role" });
    }
    
    const oldRole = await getUserRole(firebaseUid);
    await setUserRole(firebaseUid, role as Role, req.user!.uid);
    
    logAuditEvent({
      adminUserId: req.user!.uid,
      adminRole: req.user!.role,
      action: 'CHANGE_USER_ROLE',
      targetUserId: firebaseUid,
      oldValue: oldRole,
      newValue: role,
      details: `Role changed from ${oldRole || 'none'} to ${role} for user ${user.email}`
    });
    
    res.json({ success: true, userId, newRole: role, oldRole });
  } catch (error) {
    console.error("Admin set role error:", error);
    const errorMessage = error.message || error.toString();
    res.status(error.message === 'insufficient_role' ? 403 : 500).json({ 
      error: error.message === 'insufficient_role' ? 'Insufficient permissions' : errorMessage
    });
  }
});

// Get user's current role
router.get("/api/admin/users/:userId/role", devBypass('management'), requireManagementOrAbove, async (req: AuthenticatedRequest, res) => {
  try {
    const { userId } = req.params;
    
    // Look up the user to get their Firebase UID
    const user = await storage.getUser(userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    
    const role = await getUserRole(user.firebaseUid);
    console.log(`🔍 Role lookup: ${userId} (${user.firebaseUid}) -> ${role || 'none'} (by ${req.user!.uid})`);
    
    res.json({ userId, role });
  } catch (error) {
    console.error("Admin get role error:", error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Audit log endpoint for role changes
router.get("/api/admin/audit", devBypass('management'), requireManagementOrAbove, async (req: AuthenticatedRequest, res) => {
  try {
    const { limit = 50, userId } = req.query;
    
    logAuditEvent({
      adminUserId: req.user!.uid,
      adminRole: req.user!.role,
      action: 'VIEW_AUDIT_LOG',
      details: userId ? `Viewed audit log for user ${userId}` : 'Viewed general audit log'
    });
    
    const auditEvents = userId 
      ? getAuditEventsForUser(userId as string, parseInt(limit as string))
      : getRecentAuditEvents(parseInt(limit as string));
    
    res.json({ 
      auditEvents,
      total: auditEvents.length,
      requestedBy: req.user!.uid,
      requestedAt: new Date().toISOString()
    });
  } catch (error) {
    console.error("Admin audit error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
});

// Test endpoint to verify role system functionality
router.get("/api/admin/test/role-system", devBypass('analyst'), requireAnalystOrAbove, async (req: AuthenticatedRequest, res) => {
  try {
    logAuditEvent({
      adminUserId: req.user!.uid,
      adminRole: req.user!.role,
      action: 'TEST_ROLE_SYSTEM',
      details: 'Tested role-based authentication system'
    });
    
    res.json({
      success: true,
      message: 'Role-based authentication system is working',
      adminUser: {
        uid: req.user!.uid,
        role: req.user!.role,
        email: req.user!.email
      },
      systemInfo: {
        roleHierarchy: {
          owner: 4,
          management: 3,
          staff: 2,
          analyst: 1,
          pro: 0,
          user: 0
        },
        adminRoles: ['owner', 'management', 'staff', 'analyst'],
        backwardCompatibility: 'Supports legacy admin=true tokens as owner role'
      },
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    console.error("Role system test error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
});

// Owner-only admin metrics endpoint - KPI Dashboard
router.get("/api/admin/metrics", devBypass('owner'), requireOwner, async (req: AuthenticatedRequest, res) => {
  try {
    logAuditEvent({
      adminUserId: req.user!.uid,
      adminRole: req.user!.role,
      action: 'VIEW_ADMIN_METRICS',
      details: 'Accessed owner KPI dashboard metrics'
    });

    // Get user counts from database
    const { neon } = await import("@neondatabase/serverless");
    const sql = neon(process.env.DATABASE_URL!);
    
    // Free and Pro user counts
    const freeUsersResult = await sql`SELECT COUNT(*) as count FROM users WHERE is_paid = false OR is_paid IS NULL`;
    const proUsersResult = await sql`SELECT COUNT(*) as count FROM users WHERE is_paid = true`;
    const freeUsers = parseInt(freeUsersResult[0].count);
    const proUsers = parseInt(proUsersResult[0].count);
    
    // Pause metrics - count of currently paused users
    const pausedUsersResult = await sql`SELECT COUNT(*) as count FROM users WHERE subscription_status = 'paused'`;
    const pausedUsers = parseInt(pausedUsersResult[0].count);
    
    // New Pro users this month (using pro_activated_at)
    const startMonth = startOfMonth();
    const newProResult = await sql`SELECT COUNT(*) as count FROM users WHERE is_paid = true AND pro_activated_at >= ${startMonth.toISOString()}`;
    const newProThisMonth = parseInt(newProResult[0].count);

    // --- Stripe Revenue Metrics ---
    let mtdRevenue = 0;
    let ytdRevenue = 0;
    let churnThisMonthCount = 0;
    let activeSubs = proUsers; // Default fallback
    let stripeConfigured = false;
    
    try {
      const stripeClient = getStripe();
      if (stripeClient) {
        stripeConfigured = true;
        
        // MTD Revenue (Month-to-Date)
        const mtdInvoices = await stripeClient.invoices.list({
          status: "paid",
          created: { gte: Math.floor(startOfMonth().getTime() / 1000) },
          limit: 100,
        });
        mtdRevenue = mtdInvoices.data.reduce((sum, inv) => sum + (inv.total || 0), 0) / 100;

        // YTD Revenue (Year-to-Date)  
        const ytdInvoices = await stripeClient.invoices.list({
          status: "paid",
          created: { gte: Math.floor(startOfYear().getTime() / 1000) },
          limit: 100,
        });
        ytdRevenue = ytdInvoices.data.reduce((sum, inv) => sum + (inv.total || 0), 0) / 100;

        // Active subscriptions count (include both active and paused)
        const activeSubscriptions = await stripeClient.subscriptions.list({
          status: "active",
          limit: 100,
        });
        const pausedSubscriptions = await stripeClient.subscriptions.list({
          status: "paused", 
          limit: 100,
        });
        activeSubs = activeSubscriptions.data.length + pausedSubscriptions.data.length;

        // Churn this month (canceled subscriptions)
        const canceledSubs = await stripeClient.subscriptions.list({
          status: "canceled",
          created: { gte: Math.floor(startOfMonth().getTime() / 1000) },
          limit: 100,
        });
        churnThisMonthCount = canceledSubs.data.length;
      } else {
        console.warn("Stripe not configured - revenue metrics will show zeros");
      }

    } catch (stripeError) {
      console.warn("Stripe API error (falling back to database counts):", stripeError);
      stripeConfigured = false;
    }

    // Calculate churn rate percentage
    const churnThisMonthRate = proUsers > 0 
      ? Number(((churnThisMonthCount / proUsers) * 100).toFixed(1))
      : 0;

    // Free to Pro conversion rate this month
    const totalUsers = freeUsers + proUsers;
    const conversionRate = totalUsers > 0 
      ? Number(((newProThisMonth / totalUsers) * 100).toFixed(1))
      : 0;

    // Calculate real unsatisfied percentage from cancellation data
    let unsatisfiedPct = 0;
    try {
      const allCancellations = await storage.getAllCancellations();
      const thisMonthCancellations = allCancellations.filter(c => 
        c.createdAt && new Date(c.createdAt) >= startOfMonth()
      );
      
      if (thisMonthCancellations.length > 0) {
        const unsatisfiedCancellations = thisMonthCancellations.filter(c => 
          c.reason === CANCELLATION_REASONS.DIDNT_GET_VALUE
        );
        unsatisfiedPct = Number(((unsatisfiedCancellations.length / thisMonthCancellations.length) * 100).toFixed(1));
      }
    } catch (error) {
      console.warn("Failed to calculate unsatisfied percentage:", error);
      unsatisfiedPct = 0; // Fallback to 0 if calculation fails
    }

    // Get recent pauses for owner dashboard
    let recentPauses: Array<{
      id: string;
      email: string;
      months: number;
      resumeAt: Date;
      createdAt: Date;
      reason?: string;
      note?: string;
    }> = [];
    
    try {
      recentPauses = await storage.getRecentPauses(10);
    } catch (error) {
      console.warn("Failed to fetch recent pauses:", error);
    }

    const metrics = {
      mtdRevenue,
      ytdRevenue,
      activePro: proUsers,
      churnThisMonthCount,
      churnThisMonthRate,
      newProThisMonth,
      freeUsers,
      proUsers,
      pausedUsers, // New: count of paused users
      activeSubs,
      unsatisfiedPct,
      conversionRate, // Free to Pro conversion this month
      totalUsers,
      recentPauses, // New: recent pauses for owner dashboard
      stripeConfigured // Flag to indicate if Stripe data is reliable
    };

    console.log("✅ Owner KPI metrics:", metrics);
    res.json(metrics);

  } catch (error) {
    console.error("Admin metrics error:", error);
    res.status(500).json({ 
      error: "metrics_failed", 
      message: error instanceof Error ? error.message : "Failed to fetch admin metrics" 
    });
  }
});

export default router;