import type { Role } from "../../shared/schema";

let admin: any;
let isInitialized = false;

// Role hierarchy for permission checks (SN specification: owner(4) > management(3) > creator(2.5) > staff(2) > analyst(1) > pro/user(0))
const ROLE_HIERARCHY: Record<Role, number> = {
  'owner': 4,
  'management': 3,
  'manager': 3, // Compatibility alias for management
  'creator': 2.5,
  'staff': 2,
  'analyst': 1,
  'pro': 0,
  'user': 0
};

// Admin roles that have access to admin endpoints
const ADMIN_ROLES: Role[] = ['owner', 'management', 'manager', 'staff', 'analyst'];

// Creator and admin roles that have access to creator dashboard
export const CREATOR_DASHBOARD_ROLES: Role[] = ['creator', 'owner', 'management', 'manager', 'staff', 'analyst'];

// Async Firebase Admin initialization
async function initializeFirebaseAdmin() {
  if (isInitialized) return admin;
  
  try {
    // Use dynamic import to avoid require() issues in ES modules
    const firebaseAdminModule = await import("firebase-admin");
    admin = firebaseAdminModule.default || firebaseAdminModule;
    
    const creds = {
      projectId: process.env.FIREBASE_PROJECT_ID!,
      clientEmail: process.env.FIREBASE_CLIENT_EMAIL!,
      privateKey: (process.env.FIREBASE_PRIVATE_KEY || "").replace(/\\n/g, "\n"),
    };

    if (!admin.apps.length) {
      admin.initializeApp({ credential: admin.credential.cert(creds as any) });
    }
    
    isInitialized = true;
    return admin;
  } catch (error) {
    console.warn('Firebase Admin SDK not available:', error);
    return null;
  }
}

export async function verifyAdminBearer(authHeader?: string, minimumRole: Role = 'staff') {
  const adminInstance = await initializeFirebaseAdmin();
  
  if (!adminInstance) {
    throw new Error("Firebase Admin SDK not initialized");
  }
  if (!authHeader?.startsWith("Bearer ")) throw new Error("missing");
  const token = authHeader.slice(7);
  const decoded = await adminInstance.auth().verifyIdToken(token, true);
  
  // Check role-based claims first
  if (decoded.role) {
    const userRole = decoded.role as Role;
    if (!ADMIN_ROLES.includes(userRole)) {
      console.log(`User ${decoded.uid} has non-admin role: ${userRole}`);
      throw new Error("insufficient_role");
    }
    
    // Check if user has minimum required role level
    if (ROLE_HIERARCHY[userRole] < ROLE_HIERARCHY[minimumRole]) {
      console.log(`User ${decoded.uid} role ${userRole} insufficient for minimum ${minimumRole}`);
      throw new Error("insufficient_role");
    }
    
    console.log(`✅ Role-based auth success: ${decoded.uid} (${userRole})`);
    return { ...decoded, role: userRole };
  }
  
  // Backward compatibility: check legacy admin boolean claim
  if (decoded.admin === true) {
    console.log(`⚠️ Legacy admin token detected for ${decoded.uid}, treating as 'owner'`);
    return { ...decoded, role: 'owner' as Role };
  }
  
  console.log(`❌ No valid admin role or legacy admin claim for ${decoded.uid}`);
  throw new Error("not_admin");
}

/**
 * Set a user's role via Firebase custom claims
 * Only callable server-side via Firebase Admin SDK
 */
export async function setUserRole(uid: string, role: Role, setByUid?: string): Promise<void> {
  const adminInstance = await initializeFirebaseAdmin();
  
  if (!adminInstance) {
    throw new Error("Firebase Admin SDK not initialized");
  }
  
  try {
    // Set the role as a custom claim
    await adminInstance.auth().setCustomUserClaims(uid, { role });
    
    console.log(`✅ Set role '${role}' for user ${uid}${setByUid ? ` (set by ${setByUid})` : ''}`);
    
    // Optional: Remove legacy admin claim if it exists
    const user = await adminInstance.auth().getUser(uid);
    if (user.customClaims?.admin !== undefined) {
      const updatedClaims = { ...user.customClaims };
      delete updatedClaims.admin;
      updatedClaims.role = role;
      await adminInstance.auth().setCustomUserClaims(uid, updatedClaims);
      console.log(`🔄 Migrated legacy admin claim to role '${role}' for user ${uid}`);
    }
  } catch (error) {
    console.error(`❌ Failed to set role '${role}' for user ${uid}:`, error);
    throw error;
  }
}

/**
 * Get a user's current role from Firebase custom claims
 */
export async function getUserRole(uid: string): Promise<Role | null> {
  const adminInstance = await initializeFirebaseAdmin();
  
  if (!adminInstance) {
    throw new Error("Firebase Admin SDK not initialized");
  }
  
  try {
    const user = await adminInstance.auth().getUser(uid);
    const customClaims = user.customClaims || {};
    
    // Check for role-based claim first
    if (customClaims.role) {
      return customClaims.role as Role;
    }
    
    // Check for legacy admin claim
    if (customClaims.admin === true) {
      return 'owner'; // Treat legacy admin as owner
    }
    
    return null; // No role set
  } catch (error) {
    console.error(`❌ Failed to get role for user ${uid}:`, error);
    throw error;
  }
}

/**
 * Check if a user has a specific role or higher
 */
export function hasRolePermission(userRole: Role, requiredRole: Role): boolean {
  return ROLE_HIERARCHY[userRole] >= ROLE_HIERARCHY[requiredRole];
}

/**
 * Verify if a role is a valid admin role
 */
export function isAdminRole(role: Role): boolean {
  return ADMIN_ROLES.includes(role);
}

export async function getFirebaseAdmin() {
  return await initializeFirebaseAdmin();
}