import type { Role } from "../../shared/schema";

export interface AuditEvent {
  id: string;
  timestamp: string;
  adminUserId: string;
  adminRole: Role;
  action: string;
  targetUserId?: string;
  oldValue?: any;
  newValue?: any;
  details?: string;
  ipAddress?: string;
  userAgent?: string;
}

// In-memory audit log for development (could be replaced with database storage)
let auditLog: AuditEvent[] = [];
const MAX_AUDIT_ENTRIES = 1000;

/**
 * Log an audit event for admin actions
 */
export function logAuditEvent(event: Omit<AuditEvent, 'id' | 'timestamp'>): void {
  const auditEvent: AuditEvent = {
    id: generateAuditId(),
    timestamp: new Date().toISOString(),
    ...event
  };
  
  // Add to in-memory log
  auditLog.unshift(auditEvent);
  
  // Keep only last MAX_AUDIT_ENTRIES
  if (auditLog.length > MAX_AUDIT_ENTRIES) {
    auditLog = auditLog.slice(0, MAX_AUDIT_ENTRIES);
  }
  
  // Console log for immediate visibility
  console.log(`🔍 AUDIT: ${event.action} by ${event.adminUserId} (${event.adminRole})${event.targetUserId ? ` on ${event.targetUserId}` : ''}${event.details ? ` - ${event.details}` : ''}`);
  
  // TODO: In production, also write to persistent storage
  // await storage.saveAuditEvent(auditEvent);
}

/**
 * Get recent audit events
 */
export function getRecentAuditEvents(limit: number = 50): AuditEvent[] {
  return auditLog.slice(0, limit);
}

/**
 * Get audit events for a specific user
 */
export function getAuditEventsForUser(userId: string, limit: number = 20): AuditEvent[] {
  return auditLog
    .filter(event => event.targetUserId === userId || event.adminUserId === userId)
    .slice(0, limit);
}

/**
 * Generate a unique audit ID
 */
function generateAuditId(): string {
  return `audit_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
}

/**
 * Clear audit log (for testing purposes)
 */
export function clearAuditLog(): void {
  auditLog = [];
  console.log("🧹 Audit log cleared");
}