// scripts/test-opensrs.ts
// Quick OT&E sanity test for OpenSRS from IBrandBiz (no curl needed)

import 'dotenv/config';

type OpenSrsResponse = {
  is_success?: 0 | 1;
  response_text?: string;
  attributes?: Record<string, any>;
};

const BASE = process.env.OPENSRS_BASE ?? 'https://horizon.opensrs.net:55443';
const USER = process.env.OPENSRS_USER;
const KEY  = process.env.OPENSRS_KEY;
const MODE = process.env.OPENSRS_MODE ?? 'ote';

if (!USER || !KEY) {
  console.error('Missing OPENSRS_USER or OPENSRS_KEY. Set env secrets first.');
  process.exit(1);
}

async function callOpenSRS(action: string, object: string, attributes: Record<string, any>): Promise<OpenSrsResponse> {
  const body = {
    protocol: 'XCP',
    action,
    object,
    attributes,
  };

  const res = await fetch(BASE, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      'X-Username': USER,
      'X-Signature': KEY,
    },
    body: JSON.stringify(body),
  });

  const text = await res.text();
  try {
    return JSON.parse(text);
  } catch {
    throw new Error(`Non-JSON response from OpenSRS (${res.status}):\n${text}`);
  }
}

async function main() {
  const testDomain = (process.argv[2] || 'cafemojo.com').toLowerCase();

  console.log('--- OpenSRS OT&E sanity check ---');
  console.log(`Mode: ${MODE}`);
  console.log(`Base: ${BASE}`);
  console.log(`User: ${USER}`);
  console.log(`Checking domain: ${testDomain}\n`);

  try {
    const resp = await callOpenSRS('LOOKUP', 'DOMAIN', {
      domain: testDomain,
      search_type: 'domain',
    });

    if (resp.is_success !== 1) {
      console.error('OpenSRS returned an error:', resp.response_text || resp);
      process.exit(2);
    }

    // Different responses return availability under different keys; normalize best-effort:
    const attrs = resp.attributes ?? {};
    const isAvailable =
      attrs.is_available ?? attrs.domain ?? attrs.lookup ?? attrs['is-available'];

    console.log('Raw response:', JSON.stringify(resp, null, 2), '\n');

    if (typeof isAvailable !== 'undefined') {
      console.log(`✅ Availability: ${Boolean(isAvailable) ? 'AVAILABLE' : 'TAKEN'}`);
    } else {
      console.log('ℹ️  Success, but availability flag not found—inspect raw response above.');
    }

    console.log('\nDone.');
  } catch (e: any) {
    console.error('❌ OpenSRS call failed:', e?.message || e);
    process.exit(3);
  }
}

main();