import { chromium } from '@playwright/test';
import { writeFileSync } from 'fs';

const ORIGIN = process.env.SPA_ORIGIN || 'http://localhost:5000';
const CREDS = {
  email: process.env.SPA_EMAIL || null,
  password: process.env.SPA_PASSWORD || null,
};

const startPaths = [
  '/', '/pricing/business-development', '/pricing/web-services', '/pricing/addons',
  '/resources', '/faq', '/about', '/services', '/products',
  '/business-assets/logo-templates',
  '/brand-development/ai-logo-creator',
  '/business-development/brand-name-wizard',
  '/business-development/slogan-generator',
  '/business-development/ai-business-plan-developer',
  '/web-services/domains',
];

const queue = new Set(startPaths.map(p => new URL(p, ORIGIN).toString()));
const visited = new Map(); // url -> { status, from }

const isInternal = (href) => href.startsWith(ORIGIN) || href.startsWith('/');
const norm = (href) => href.startsWith('http') ? href : new URL(href, ORIGIN).toString();

console.log(`🔍 Starting SPA audit on ${ORIGIN}...`);
console.log('This may take a few minutes...\n');

const browser = await chromium.launch();
const ctx = await browser.newContext();
const page = await ctx.newPage();

// optional login
if (CREDS.email && CREDS.password) {
  console.log('🔐 Attempting to log in...');
  try {
    await page.goto(new URL('/signup', ORIGIN).toString(), { waitUntil: 'domcontentloaded' });
    await page.fill('input[type="email"]', CREDS.email);
    await page.fill('input[type="password"]', CREDS.password);
    await page.click('button:has-text("Log in"), button:has-text("Sign in")');
    await page.waitForLoadState('networkidle', { timeout: 10000 }).catch(()=>{});
    console.log('✅ Logged in successfully\n');
  } catch (e) {
    console.log('⚠️  Login failed or not required\n');
  }
}

async function visit(url, from=null) {
  if (visited.has(url)) return;
  
  try {
    console.log(`Visiting: ${url}`);
    await page.goto(url, { waitUntil: 'domcontentloaded', timeout: 15000 });
    
    // Wait a bit for any client-side rendering
    await page.waitForTimeout(1000);
    
    // detect client 404 by data-not-found attribute or "Page not found" text
    const notFound = await page.locator('[data-not-found], text="Page not found"').first().isVisible().catch(()=>false);
    visited.set(url, { status: notFound ? 'not-found' : 'ok', from });

    if (notFound) {
      console.log(`  ❌ 404 detected`);
      return;
    }
    
    console.log(`  ✅ OK`);

    // Find all internal links on this page
    const links = await page.locator('a[href]').all();
    for (const a of links) {
      const href = await a.getAttribute('href');
      if (!href) continue;
      if (!isInternal(href)) continue;
      if (href.startsWith('#')) continue; // skip anchors
      if (href.startsWith('mailto:')) continue;
      if (href.startsWith('tel:')) continue;
      
      const abs = norm(href);
      if (!visited.has(abs) && !queue.has(abs)) {
        queue.add(abs);
      }
    }
  } catch (e) {
    console.log(`  ⚠️  Blocked or error: ${e.message}`);
    visited.set(url, { status: 'blocked', from, error: e.message });
  }
}

// Crawl all pages
while (queue.size) {
  const [next] = queue;
  queue.delete(next);
  await visit(next, null);
}

const report = Array.from(visited.entries()).map(([url, meta]) => ({ url, ...meta }));
writeFileSync('spa-link-report.json', JSON.stringify({ origin: ORIGIN, total: report.length, items: report }, null, 2));

console.log(`\n🔎 SPA Audit Complete for ${ORIGIN}`);
console.log(`Total pages checked: ${report.length}`);

const broken = report.filter(r => r.status === 'not-found');
const blocked = report.filter(r => r.status === 'blocked');
const ok = report.filter(r => r.status === 'ok');

console.log(`✅ OK: ${ok.length}`);
console.log(`❌ 404s: ${broken.length}`);
console.log(`⚠️  Blocked/Errors: ${blocked.length}`);

if (broken.length > 0) {
  console.log('\nBroken Pages (404):');
  broken.slice(0, 25).forEach((b,i)=>console.log(`${i+1}. ${b.url}${b.from ? ` (from: ${b.from})` : ''}`));
  if (broken.length > 25) {
    console.log(`\n... and ${broken.length - 25} more (see spa-link-report.json)`);
  }
}

if (blocked.length > 0) {
  console.log('\nBlocked/Error Pages:');
  blocked.slice(0, 10).forEach((b,i)=>console.log(`${i+1}. ${b.url} - ${b.error || 'unknown error'}`));
}

console.log('\nFull report saved to spa-link-report.json\n');

await browser.close();
