#!/usr/bin/env node
/**
 * One-time migration script: logo_templates → templates
 * 
 * USAGE:
 *   node scripts/migrate-logo-templates.js [--dry-run] [--force]
 * 
 * REQUIREMENTS:
 *   - FIREBASE_SERVICE_ACCOUNT_KEY environment variable with service account JSON
 *   - Run during low-traffic window
 *   - Backup/export collections before running
 */

import admin from "firebase-admin";
import { readFileSync } from "fs";

// Initialize Firebase Admin
const serviceAccountKey = process.env.FIREBASE_SERVICE_ACCOUNT_KEY;
if (!serviceAccountKey) {
  console.error("❌ FIREBASE_SERVICE_ACCOUNT_KEY environment variable required");
  process.exit(1);
}

admin.initializeApp({
  credential: admin.credential.cert(JSON.parse(serviceAccountKey)),
});

const db = admin.firestore();

// Parse command line arguments
const isDryRun = process.argv.includes('--dry-run');
const isForced = process.argv.includes('--force');

console.log(`🚀 Starting logo_templates → templates migration`);
console.log(`📋 Mode: ${isDryRun ? 'DRY RUN' : 'LIVE MIGRATION'}`);
console.log(`⚠️  Force overwrite: ${isForced ? 'YES' : 'NO'}`);

async function runMigration() {
  try {
    // Step 1: Read source collection
    console.log("\n📖 Reading logo_templates collection...");
    const sourceSnapshot = await db.collection("logo_templates").get();
    console.log(`📊 Found ${sourceSnapshot.size} documents in logo_templates`);

    if (sourceSnapshot.empty) {
      console.log("✅ No documents to migrate");
      return;
    }

    // Step 2: Check for conflicts in destination
    console.log("\n🔍 Checking for conflicts in templates collection...");
    const conflicts = [];
    const sourceIds = sourceSnapshot.docs.map(doc => doc.id);
    
    // Check existing docs in batches of 10 (Firestore 'in' query limit)
    for (let i = 0; i < sourceIds.length; i += 10) {
      const batch = sourceIds.slice(i, i + 10);
      const existingDocs = await db.collection("templates")
        .where(admin.firestore.FieldPath.documentId(), 'in', batch)
        .get();
      
      existingDocs.forEach(doc => {
        conflicts.push(doc.id);
      });
    }

    if (conflicts.length > 0 && !isForced) {
      console.log(`⚠️  Found ${conflicts.length} conflicting document IDs:`);
      conflicts.forEach(id => console.log(`   - ${id}`));
      console.log("🛑 Use --force to overwrite existing documents");
      return;
    }

    // Step 3: Prepare migration batches (max 500 per batch)
    const BATCH_SIZE = 500;
    const docs = sourceSnapshot.docs;
    let migratedCount = 0;

    console.log(`\n📦 Processing ${docs.length} documents in batches of ${BATCH_SIZE}...`);

    for (let i = 0; i < docs.length; i += BATCH_SIZE) {
      const batchDocs = docs.slice(i, i + BATCH_SIZE);
      console.log(`\n📋 Processing batch ${Math.floor(i / BATCH_SIZE) + 1} (${batchDocs.length} docs)...`);

      if (!isDryRun) {
        const batch = db.batch();

        batchDocs.forEach(doc => {
          const data = doc.data();
          const destRef = db.collection("templates").doc(doc.id);
          
          // Add migration metadata
          const migrationData = {
            ...data,
            migratedFrom: "logo_templates",
            migratedAt: admin.firestore.FieldValue.serverTimestamp(),
            updatedAt: admin.firestore.FieldValue.serverTimestamp()
          };

          batch.set(destRef, migrationData, { merge: true });
        });

        await batch.commit();
        console.log(`✅ Committed batch ${Math.floor(i / BATCH_SIZE) + 1}`);
      } else {
        console.log(`📝 [DRY RUN] Would migrate batch ${Math.floor(i / BATCH_SIZE) + 1}`);
      }

      migratedCount += batchDocs.length;
    }

    // Step 4: Verification
    if (!isDryRun) {
      console.log("\n🔍 Verifying migration...");
      const destSnapshot = await db.collection("templates").get();
      const migratedDocs = destSnapshot.docs.filter(doc => 
        doc.data().migratedFrom === "logo_templates"
      );
      
      console.log(`📊 Verification: ${migratedDocs.length}/${sourceSnapshot.size} documents migrated`);
      
      if (migratedDocs.length === sourceSnapshot.size) {
        console.log("✅ Migration completed successfully!");
      } else {
        console.log("⚠️  Migration incomplete - manual verification required");
      }
    }

    console.log(`\n🎉 Migration summary:`);
    console.log(`   📁 Source documents: ${sourceSnapshot.size}`);
    console.log(`   📦 Processed: ${migratedCount}`);
    console.log(`   ⚠️  Conflicts: ${conflicts.length}`);
    console.log(`   🔧 Mode: ${isDryRun ? 'DRY RUN' : 'LIVE'}`);

    if (!isDryRun) {
      console.log(`\n📋 Next steps:`);
      console.log(`   1. Update Firestore rules for 'templates' collection`);
      console.log(`   2. Create composite indexes: status, isPrivate, updatedAt`);
      console.log(`   3. Test application with single collection`);
      console.log(`   4. Remove dual collection queries from code`);
      console.log(`   5. Archive/delete 'logo_templates' collection`);
    }

  } catch (error) {
    console.error("❌ Migration failed:", error);
    process.exit(1);
  }
}

// Confirmation prompt for live migrations
if (!isDryRun && !isForced) {
  console.log("\n⚠️  LIVE MIGRATION WARNING:");
  console.log("   - Ensure you have backups of both collections");
  console.log("   - Run during low-traffic window");
  console.log("   - Monitor application health during migration");
  console.log("\n🔄 Run with --dry-run first to test, then --force to confirm");
  
  const readline = require('readline').createInterface({
    input: process.stdin,
    output: process.stdout
  });
  
  readline.question('\nProceed with live migration? (yes/no): ', (answer) => {
    readline.close();
    if (answer.toLowerCase() === 'yes') {
      runMigration();
    } else {
      console.log("🛑 Migration cancelled");
    }
  });
} else {
  runMigration();
}