// Crawl the running site and write a JSON + pretty report.
import { LinkChecker } from 'linkinator';
import { writeFileSync } from 'fs';

const origin = process.env.LINK_AUDIT_ORIGIN || 'http://localhost:5000';
const checker = new LinkChecker();

console.log(`🔍 Starting link audit on ${origin}...`);
console.log('This may take a few minutes...\n');

const result = await checker.check({
  path: origin,
  recurse: true,
  linksToIgnore: [
    // ignore external & mail links
    'mailto:*',
    'tel:*',
    'https://*',
    'http://localhost:*',
  ],
  // Only consider same-origin
  maxDepth: 4
});

const broken = result.links.filter(l =>
  (l.state === 'BROKEN') ||
  (l.status === 404) ||
  (l.status === 0 && l.failureReason) // fetch failed
);

const summary = {
  origin,
  totalLinks: result.links.length,
  brokenCount: broken.length,
  brokenByUrl: broken.map(l => ({
    url: l.url,
    status: l.status,
    state: l.state,
    failure: l.failureReason || null,
    source: l.parent || null
  }))
};

writeFileSync('./link-report.json', JSON.stringify(summary, null, 2));
console.log('\n🔎 Link Audit Summary');
console.log(`Origin: ${origin}`);
console.log(`Total links scanned: ${summary.totalLinks}`);
console.log(`❌ Broken/404: ${summary.brokenCount}`);

if (summary.brokenCount > 0) {
  console.log('\nBroken Links:');
  summary.brokenByUrl.slice(0, 25).forEach((b, i) => {
    console.log(`${i+1}. [${b.status || b.state}] ${b.url}`);
    if (b.source) {
      console.log(`   Found on: ${b.source}`);
    }
  });
  if (summary.brokenCount > 25) {
    console.log(`\n... and ${summary.brokenCount - 25} more (see link-report.json)`);
  }
} else {
  console.log('✅ No broken links found!');
}

console.log('\nFull report saved to link-report.json\n');
