#!/usr/bin/env ts-node
import fs from "fs";
import path from "path";
import { DOC_DIR, PREV_DIR, loadManifest, saveManifest, upsert, BPItem } from "../client/src/lib/bp-io";

const today = new Date().toISOString().slice(0,10);

function titleCaseFromId(id: string) {
  return id.replace(/\-+/g," ").replace(/\b\w/g, c => c.toUpperCase());
}

function findDocForId(id: string): string | null {
  const docs = fs.readdirSync(DOC_DIR).filter(f => /\.docx$/i.test(f));
  // Prefer filename that contains the id
  const hit = docs.find(f => f.toLowerCase().includes(id));
  return hit || docs.find(() => true) || null;
}

(async () => {
  const manifest = loadManifest();

  const previews = fs.readdirSync(PREV_DIR).filter(f => /\.(png|jpg|jpeg)$/i.test(f));
  let added = 0;

  for (const prev of previews) {
    const id = prev.replace(/\.(png|jpe?g)$/i, "").toLowerCase();
    const docName = findDocForId(id);
    if (!docName) { console.warn(`No DOCX found for id=${id} (preview=${prev})`); continue; }

    // Defaults
    let item: BPItem = {
      id,
      name: `Business Plan — ${titleCaseFromId(id)}`,
      category: "General",
      tags: [],
      previewUrl: `/templates/business-plan/previews/${prev}`,
      docxUrl: `/templates/business-plan/docs/${docName}`,
      updatedAt: today,
      sections: [
        "Executive Summary","Market Analysis","Products & Services",
        "Marketing & Sales","Operations Plan","Organization & Management","Financial Plan","Appendices"
      ]
    };

    // Optional sidecar
    const metaPath = path.join(PREV_DIR, `${id}.meta.json`);
    if (fs.existsSync(metaPath)) {
      try {
        const meta = JSON.parse(fs.readFileSync(metaPath, "utf8"));
        item = { ...item, ...meta, id }; // never override ID
      } catch (e) {
        console.warn(`Invalid meta JSON for ${id}:`, (e as any).message);
      }
    }

    upsert(manifest.items, item);
    added++;
  }

  saveManifest(manifest);
  console.log(`✓ Manifest updated. Items: ${manifest.items.length} (processed ${added} previews).`);
})();