// Cloud Function for auto-generating template previews following SuperNova's specification
import { onObjectFinalized } from "firebase-functions/v2/storage";
import { Storage } from "@google-cloud/storage";
import sharp from "sharp";

const storage = new Storage();
const BUCKET = process.env.FUNCTIONS_EMULATOR ? "demo-bucket" : process.env.GCLOUD_PROJECT + ".firebasestorage.app";

export const makeTemplatePreview = onObjectFinalized(
  { bucket: BUCKET, memory: "1GiB", region: "us-central1" },
  async (event) => {
    const file = event.data;
    if (!file || !file.name) return;

    const path = file.name; // e.g. templates/slug/v1/art.svg
    if (!path.endsWith("/art.svg")) return; // only process template SVGs

    console.log(`Processing template SVG: ${path}`);

    try {
      const bucket = storage.bucket(file.bucket);
      const [buf] = await bucket.file(path).download();

      // Render SVG to PNG (transparent background)
      // Output size can be tuned; 1600 wide is a good crisp preview baseline
      const png = await sharp(buf, { density: 300 })
        .png({ compressionLevel: 9, adaptiveFiltering: true })
        .resize({ width: 1600 })
        .toBuffer();

      const previewPath = path.replace("art.svg", "preview.png");
      await bucket.file(previewPath).save(png, {
        contentType: "image/png",
        public: true,
        metadata: {
          cacheControl: "public, max-age=31536000, s-maxage=31536000"
        }
      });
      
      console.log("Preview generated:", previewPath);
    } catch (error) {
      console.error("Failed to generate preview:", error);
    }
  }
);