// Normalize SVG by wrapping visual content in #logo-root group
export function normalizeSvgWithLogoRoot(svg: string): string {
  if (!svg) return '';
  
  try {
    const doc = new DOMParser().parseFromString(svg, "image/svg+xml");
    
    if (doc.querySelector("parsererror")) {
      console.warn("SVG parsing failed in normalization");
      return svg;
    }
    
    const svgElement = doc.documentElement;
    
    // Check if #logo-root already exists
    let logoRoot = svgElement.querySelector('#logo-root');
    
    if (!logoRoot) {
      // Create #logo-root group
      logoRoot = doc.createElementNS('http://www.w3.org/2000/svg', 'g');
      logoRoot.setAttribute('id', 'logo-root');
      
      // Move all children except <defs>, non-rendering elements, and structural paint servers into #logo-root
      const children = Array.from(svgElement.children);
      // Keep structural/definition elements at SVG root so they can be referenced via url(#id)
      const structuralTags = ['defs', 'metadata', 'title', 'desc', 'mask', 'clipPath', 'clippath', 'pattern', 'filter', 'linearGradient', 'lineargradient', 'radialGradient', 'radialgradient', 'symbol', 'marker', 'style'];
      
      children.forEach(child => {
        const tagName = child.tagName.toLowerCase();
        if (!structuralTags.includes(tagName)) {
          logoRoot!.appendChild(child.cloneNode(true));
          child.remove();
        }
      });
      
      // Append #logo-root after any <defs>
      const defs = svgElement.querySelector('defs');
      if (defs && defs.nextSibling) {
        svgElement.insertBefore(logoRoot, defs.nextSibling);
      } else {
        svgElement.appendChild(logoRoot);
      }
    }
    
    // A) Strip hard sizing from <svg> (SN's spec)
    svgElement.removeAttribute('width');
    svgElement.removeAttribute('height');
    
    // Set preserveAspectRatio for proper scaling (SN's spec)
    svgElement.setAttribute('preserveAspectRatio', 'xMidYMid meet');
    
    // B) Ensure viewBox exists - compute from bbox if missing (SN's spec)
    if (!svgElement.hasAttribute('viewBox')) {
      // Create temporary container to measure bbox
      const tempDiv = document.createElement('div');
      tempDiv.style.position = 'absolute';
      tempDiv.style.visibility = 'hidden';
      document.body.appendChild(tempDiv);
      
      const tempSvg = doc.documentElement.cloneNode(true) as SVGSVGElement;
      tempDiv.appendChild(tempSvg);
      
      try {
        const logoRootInTemp = tempSvg.querySelector('#logo-root') as SVGGElement;
        if (logoRootInTemp) {
          const bbox = logoRootInTemp.getBBox();
          svgElement.setAttribute('viewBox', `${bbox.x} ${bbox.y} ${bbox.width} ${bbox.height}`);
        } else {
          // Fallback if no logo-root
          svgElement.setAttribute('viewBox', '0 0 100 100');
        }
      } catch (e) {
        // Fallback viewBox if bbox fails
        svgElement.setAttribute('viewBox', '0 0 100 100');
      }
      
      document.body.removeChild(tempDiv);
    }
    
    return new XMLSerializer().serializeToString(doc);
  } catch (error) {
    console.warn("SVG normalization failed:", error);
    return svg;
  }
}

export function applySvgColor(svg: string, colorSwatches: { primary: string; secondary: string; accent: string }): string {
  if (!svg) return '';
  
  try {
    // Use DOM parsing for more reliable color replacement
    const doc = new DOMParser().parseFromString(svg, "image/svg+xml");
    
    if (doc.querySelector("parsererror")) {
      console.warn("SVG parsing failed, falling back to string replacement");
      return applySvgColorFallback(svg, colorSwatches);
    }
    
    // First pass: detect all unique colors in the SVG and count their usage
    const colorUsage = new Map<string, number>();
    const classToColor = new Map<string, string>(); // Map CSS classes to their colors
    
    // Extract colors from <style> blocks
    const styleElements = doc.querySelectorAll('style');
    styleElements.forEach(styleEl => {
      const styleText = styleEl.textContent || '';
      // Match .className { fill: #color; } or .className { stroke: #color; }
      const classMatches = styleText.matchAll(/\.([^\s{]+)\s*\{[^}]*(?:fill|stroke)\s*:\s*(#[0-9a-fA-F]{3,8}|rgb[^;]+|hsl[^;]+)/gi);
      for (const match of classMatches) {
        const className = match[1];
        const color = match[2];
        if (!classToColor.has(className)) {
          classToColor.set(className, color);
        }
      }
    });
    
    const getAllColors = (element: Element) => {
      const fill = element.getAttribute('fill');
      const stroke = element.getAttribute('stroke');
      const className = element.getAttribute('class');
      
      // Check if element uses a class with a color
      if (className && classToColor.has(className)) {
        const color = classToColor.get(className)!;
        colorUsage.set(color, (colorUsage.get(color) || 0) + 1);
      }
      
      if (fill && fill !== 'none' && !fill.startsWith('url(') && !fill.startsWith('var(')) {
        if (fill.match(/^#[0-9a-fA-F]{3,8}$/) || fill.match(/^rgb/) || fill.match(/^hsl/)) {
          colorUsage.set(fill, (colorUsage.get(fill) || 0) + 1);
        }
      }
      
      if (stroke && stroke !== 'none' && !stroke.startsWith('url(') && !stroke.startsWith('var(')) {
        if (stroke.match(/^#[0-9a-fA-F]{3,8}$/) || stroke.match(/^rgb/) || stroke.match(/^hsl/)) {
          colorUsage.set(stroke, (colorUsage.get(stroke) || 0) + 1);
        }
      }
      
      Array.from(element.children).forEach(getAllColors);
    };
    getAllColors(doc.documentElement);
    
    // Sort colors by usage (most used first)
    const sortedColors = Array.from(colorUsage.entries())
      .sort((a, b) => b[1] - a[1])
      .map(entry => entry[0]);
    
    // Map the most used colors to roles (primary = most used, secondary = 2nd, accent = 3rd)
    const colorToRole = new Map<string, 'primary' | 'secondary' | 'accent'>();
    if (sortedColors.length >= 1) colorToRole.set(sortedColors[0], 'primary');
    if (sortedColors.length >= 2) colorToRole.set(sortedColors[1], 'secondary');
    if (sortedColors.length >= 3) colorToRole.set(sortedColors[2], 'accent');
    
    // Map elements to their color roles based on data-role, id, class, OR original color
    const getRoleForElement = (element: Element, colorValue: string): 'primary' | 'secondary' | 'accent' => {
      // Check data-role attribute
      const dataRole = element.getAttribute('data-role');
      if (dataRole === 'primary' || dataRole === 'secondary' || dataRole === 'accent') {
        return dataRole;
      }
      
      // Check id for role hints
      const id = element.getAttribute('id')?.toLowerCase() || '';
      if (id.includes('primary')) return 'primary';
      if (id.includes('secondary')) return 'secondary';
      if (id.includes('accent')) return 'accent';
      
      // Check class for role hints
      const className = element.getAttribute('class')?.toLowerCase() || '';
      if (className.includes('primary')) return 'primary';
      if (className.includes('secondary')) return 'secondary';
      if (className.includes('accent')) return 'accent';
      
      // Fallback: use color-based mapping
      return colorToRole.get(colorValue) || 'primary';
    };
    
    // Recursively process all elements to convert colors to CSS variables
    const processElement = (element: Element) => {
      // Handle elements with CSS classes that have colors
      const className = element.getAttribute('class');
      if (className && classToColor.has(className)) {
        const color = classToColor.get(className)!;
        const role = getRoleForElement(element, color);
        const cssVar = `var(--logo-${role})`;
        // Add inline style to override the class color
        const existingStyle = element.getAttribute('style') || '';
        element.setAttribute('style', `${existingStyle}; fill: ${cssVar};`.trim());
      }
      
      // Handle fill attribute
      if (element.hasAttribute('fill')) {
        const fill = element.getAttribute('fill');
        if (fill && fill !== 'none' && !fill.startsWith('url(') && !fill.startsWith('var(')) {
          // Convert solid colors to CSS variable
          if (fill.match(/^#[0-9a-fA-F]{3,8}$/) || fill.match(/^rgb/) || fill.match(/^hsl/)) {
            const role = getRoleForElement(element, fill);
            const cssVar = `var(--logo-${role})`;
            element.setAttribute('fill', cssVar);
          }
        }
      }
      
      // Handle stroke attribute
      if (element.hasAttribute('stroke')) {
        const stroke = element.getAttribute('stroke');
        if (stroke && stroke !== 'none' && !stroke.startsWith('url(') && !stroke.startsWith('var(')) {
          if (stroke.match(/^#[0-9a-fA-F]{3,8}$/) || stroke.match(/^rgb/) || stroke.match(/^hsl/)) {
            const role = getRoleForElement(element, stroke);
            const cssVar = `var(--logo-${role})`;
            element.setAttribute('stroke', cssVar);
          }
        }
      }
      
      // Handle inline styles
      if (element.hasAttribute('style')) {
        let style = element.getAttribute('style') || '';
        // For inline styles, we need to extract and replace each color individually
        // This is a simplified approach - replace with primary for now
        const fillMatch = style.match(/fill\s*:\s*(#[0-9a-fA-F]{3,8}|rgb[^;]+|hsl[^;]+)/i);
        if (fillMatch) {
          const role = getRoleForElement(element, fillMatch[1]);
          style = style.replace(/fill\s*:\s*#[0-9a-fA-F]{3,8}/gi, `fill: var(--logo-${role})`);
          style = style.replace(/fill\s*:\s*rgb[^;]+/gi, `fill: var(--logo-${role})`);
          style = style.replace(/fill\s*:\s*hsl[^;]+/gi, `fill: var(--logo-${role})`);
        }
        
        const strokeMatch = style.match(/stroke\s*:\s*(#[0-9a-fA-F]{3,8}|rgb[^;]+|hsl[^;]+)/i);
        if (strokeMatch) {
          const role = getRoleForElement(element, strokeMatch[1]);
          style = style.replace(/stroke\s*:\s*#[0-9a-fA-F]{3,8}/gi, `stroke: var(--logo-${role})`);
          style = style.replace(/stroke\s*:\s*rgb[^;]+/gi, `stroke: var(--logo-${role})`);
          style = style.replace(/stroke\s*:\s*hsl[^;]+/gi, `stroke: var(--logo-${role})`);
        }
        
        element.setAttribute('style', style);
      }
      
      // Process child elements
      Array.from(element.children).forEach(processElement);
    };
    
    // Start processing from the root SVG element
    const svgElement = doc.documentElement;
    processElement(svgElement);
    
    // Set CSS variables on #logo-root group (or fallback to SVG root if not present)
    const logoRoot = svgElement.querySelector('#logo-root');
    const targetElement = logoRoot || svgElement;
    
    const existingStyle = targetElement.getAttribute('style') || '';
    const newStyle = existingStyle + `; --logo-primary: ${colorSwatches.primary}; --logo-secondary: ${colorSwatches.secondary}; --logo-accent: ${colorSwatches.accent};`;
    targetElement.setAttribute('style', newStyle);
    
    return new XMLSerializer().serializeToString(doc);
  } catch (error) {
    console.warn("DOM-based SVG color replacement failed:", error);
    return applySvgColorFallback(svg, colorSwatches);
  }
}

// Fallback string-based replacement for when DOM parsing fails
function applySvgColorFallback(svg: string, colorSwatches: { primary: string; secondary: string; accent: string }): string {
  let s = svg;
  
  // Convert explicit colors to var(--logo-primary), but preserve "none", gradients, and CSS vars
  s = s.replace(/fill="(?!none|url\(|var\()[^"]*"/gi, 'fill="var(--logo-primary)"');
  s = s.replace(/stroke="(?!none|url\(|var\()[^"]*"/gi, 'stroke="var(--logo-primary)"');
  
  // Convert colors in inline styles to var(--logo-primary)
  s = s.replace(/style="([^"]*)"/gi, (match, styles) => {
    let newStyles = styles;
    // Convert explicit fill colors to var(--logo-primary)
    newStyles = newStyles.replace(/fill\s*:\s*#[0-9a-fA-F]{3,8}/gi, 'fill: var(--logo-primary)');
    newStyles = newStyles.replace(/fill\s*:\s*rgb[^;]+/gi, 'fill: var(--logo-primary)');
    newStyles = newStyles.replace(/fill\s*:\s*hsl[^;]+/gi, 'fill: var(--logo-primary)');
    // Convert explicit stroke colors to var(--logo-primary)
    newStyles = newStyles.replace(/stroke\s*:\s*#[0-9a-fA-F]{3,8}/gi, 'stroke: var(--logo-primary)');
    newStyles = newStyles.replace(/stroke\s*:\s*rgb[^;]+/gi, 'stroke: var(--logo-primary)');
    newStyles = newStyles.replace(/stroke\s*:\s*hsl[^;]+/gi, 'stroke: var(--logo-primary)');
    return `style="${newStyles}"`;
  });
  
  // Inject CSS variables on outer SVG
  if (/style\s*=\s*"/.test(s)) {
    s = s.replace(/style\s*=\s*"([^"]*)"/i, (match, styles) => {
      return `style="${styles}; --logo-primary: ${colorSwatches.primary}; --logo-secondary: ${colorSwatches.secondary}; --logo-accent: ${colorSwatches.accent};"`;
    });
  } else {
    s = s.replace(/<svg([^>]*)>/, (match, attributes) => {
      return `<svg${attributes} style="--logo-primary: ${colorSwatches.primary}; --logo-secondary: ${colorSwatches.secondary}; --logo-accent: ${colorSwatches.accent};">`;
    });
  }
  
  return s;
}

// Function to resolve currentColor and CSS variables to actual hex values for export
export function resolveSvgColorsForExport(svg: string, colorSwatches: { primary: string; secondary: string; accent: string }): string {
  if (!svg) return '';
  
  try {
    // Use DOM parsing for more reliable resolution
    const doc = new DOMParser().parseFromString(svg, "image/svg+xml");
    
    if (doc.querySelector("parsererror")) {
      console.warn("SVG parsing failed for export resolution, falling back to string replacement");
      return resolveSvgColorsForExportFallback(svg, colorSwatches);
    }
    
    // Recursively process all elements to resolve colors
    const processElement = (element: Element) => {
      // Handle fill attribute - resolve CSS variables
      if (element.hasAttribute('fill')) {
        const fill = element.getAttribute('fill');
        if (fill === 'var(--logo-primary)') {
          element.setAttribute('fill', colorSwatches.primary);
        } else if (fill === 'var(--logo-secondary)') {
          element.setAttribute('fill', colorSwatches.secondary);
        } else if (fill === 'var(--logo-accent)') {
          element.setAttribute('fill', colorSwatches.accent);
        }
      }
      
      // Handle stroke attribute - resolve CSS variables
      if (element.hasAttribute('stroke')) {
        const stroke = element.getAttribute('stroke');
        if (stroke === 'var(--logo-primary)') {
          element.setAttribute('stroke', colorSwatches.primary);
        } else if (stroke === 'var(--logo-secondary)') {
          element.setAttribute('stroke', colorSwatches.secondary);
        } else if (stroke === 'var(--logo-accent)') {
          element.setAttribute('stroke', colorSwatches.accent);
        }
      }
      
      // Handle inline styles - resolve CSS variables
      if (element.hasAttribute('style')) {
        let style = element.getAttribute('style') || '';
        // Resolve CSS variables in styles
        style = style.replace(/fill\s*:\s*var\(--logo-primary\)/gi, `fill: ${colorSwatches.primary}`);
        style = style.replace(/fill\s*:\s*var\(--logo-secondary\)/gi, `fill: ${colorSwatches.secondary}`);
        style = style.replace(/fill\s*:\s*var\(--logo-accent\)/gi, `fill: ${colorSwatches.accent}`);
        style = style.replace(/stroke\s*:\s*var\(--logo-primary\)/gi, `stroke: ${colorSwatches.primary}`);
        style = style.replace(/stroke\s*:\s*var\(--logo-secondary\)/gi, `stroke: ${colorSwatches.secondary}`);
        style = style.replace(/stroke\s*:\s*var\(--logo-accent\)/gi, `stroke: ${colorSwatches.accent}`);
        element.setAttribute('style', style);
      }
      
      // Process child elements
      Array.from(element.children).forEach(processElement);
    };
    
    // Start processing from the root SVG element
    const svgElement = doc.documentElement;
    processElement(svgElement);
    
    // Remove CSS variables from #logo-root (or SVG root) since they're now resolved
    const logoRoot = svgElement.querySelector('#logo-root');
    const targetElement = logoRoot || svgElement;
    
    const existingStyle = targetElement.getAttribute('style') || '';
    const cleanedStyle = existingStyle
      .replace(/--logo-primary\s*:\s*[^;]+;?/gi, '')
      .replace(/--logo-secondary\s*:\s*[^;]+;?/gi, '')
      .replace(/--logo-accent\s*:\s*[^;]+;?/gi, '')
      .replace(/--secondary\s*:\s*[^;]+;?/gi, '')
      .replace(/--accent\s*:\s*[^;]+;?/gi, '')
      .replace(/color\s*:\s*[^;]+;?/gi, '')
      .replace(/;\s*;/g, ';')
      .replace(/^;\s*|;\s*$/g, '');
    
    if (cleanedStyle) {
      targetElement.setAttribute('style', cleanedStyle);
    } else {
      targetElement.removeAttribute('style');
    }
    
    return new XMLSerializer().serializeToString(doc);
  } catch (error) {
    console.warn("DOM-based SVG color resolution failed:", error);
    return resolveSvgColorsForExportFallback(svg, colorSwatches);
  }
}

// Fallback string-based replacement for export resolution
function resolveSvgColorsForExportFallback(svg: string, colorSwatches: { primary: string; secondary: string; accent: string }): string {
  let s = svg;
  
  // Resolve CSS variables in attributes
  s = s.replace(/fill="var\(--logo-primary\)"/gi, `fill="${colorSwatches.primary}"`);
  s = s.replace(/fill="var\(--logo-secondary\)"/gi, `fill="${colorSwatches.secondary}"`);
  s = s.replace(/fill="var\(--logo-accent\)"/gi, `fill="${colorSwatches.accent}"`);
  s = s.replace(/stroke="var\(--logo-primary\)"/gi, `stroke="${colorSwatches.primary}"`);
  s = s.replace(/stroke="var\(--logo-secondary\)"/gi, `stroke="${colorSwatches.secondary}"`);
  s = s.replace(/stroke="var\(--logo-accent\)"/gi, `stroke="${colorSwatches.accent}"`);
  
  // Resolve colors in inline styles
  s = s.replace(/style="([^"]*)"/gi, (match, styles) => {
    let newStyles = styles;
    // Resolve CSS variables
    newStyles = newStyles.replace(/fill\s*:\s*var\(--logo-primary\)/gi, `fill: ${colorSwatches.primary}`);
    newStyles = newStyles.replace(/fill\s*:\s*var\(--logo-secondary\)/gi, `fill: ${colorSwatches.secondary}`);
    newStyles = newStyles.replace(/fill\s*:\s*var\(--logo-accent\)/gi, `fill: ${colorSwatches.accent}`);
    newStyles = newStyles.replace(/stroke\s*:\s*var\(--logo-primary\)/gi, `stroke: ${colorSwatches.primary}`);
    newStyles = newStyles.replace(/stroke\s*:\s*var\(--logo-secondary\)/gi, `stroke: ${colorSwatches.secondary}`);
    newStyles = newStyles.replace(/stroke\s*:\s*var\(--logo-accent\)/gi, `stroke: ${colorSwatches.accent}`);
    return `style="${newStyles}"`;
  });
  
  // Remove CSS variables from root SVG style
  s = s.replace(/style="([^"]*)"/i, (match, styles) => {
    let cleanedStyles = styles
      .replace(/--logo-primary\s*:\s*[^;]+;?/gi, '')
      .replace(/--logo-secondary\s*:\s*[^;]+;?/gi, '')
      .replace(/--logo-accent\s*:\s*[^;]+;?/gi, '')
      .replace(/;\s*;/g, ';')
      .replace(/^;\s*|;\s*$/g, '');
    
    if (cleanedStyles) {
      return `style="${cleanedStyles}"`;
    } else {
      return '';
    }
  });
  
  return s;
}

export function sanitizeSvgClient(svg: string) {
  try {
    const doc = new DOMParser().parseFromString(svg, "image/svg+xml");

    const allowedTags = new Set([
      "svg","g","path","rect","circle","ellipse","polygon","polyline","line",
      "defs","clipPath","mask","use","symbol","title","desc"
    ]);

    const allowedAttrs = new Set([
      "viewBox","width","height","x","y","cx","cy","r","rx","ry",
      "d","points","transform","style","class","color","xmlns","version",
      "fill","fill-rule","stroke","stroke-width","stroke-linecap","stroke-linejoin","stroke-miterlimit","opacity",
      "clip-path","mask","id","href","xlink:href","role","aria-hidden","focusable"
    ]);

    const walk = (el: Element) => {
      if (!allowedTags.has(el.tagName)) { el.remove(); return; }
      Array.from(el.attributes).forEach(a => {
        const n = a.name;
        if (n.startsWith("on")) el.removeAttribute(n);
        else if (!allowedAttrs.has(n) && !n.startsWith("aria-")) el.removeAttribute(n);
      });
      Array.from(el.children).forEach(ch => walk(ch as Element));
    };

    walk(doc.documentElement);
    return new XMLSerializer().serializeToString(doc);
  } catch {
    return svg; // fail open: better to render than disappear
  }
}