// src/utils/pngLogo.ts

// Call your backend to generate a photorealistic PNG with transparent background
export async function generateTransparentIconPNG(opts: {
  prompt: string;
  size?: number;        // 512, 768, 1024
  guidance?: number;    // optional, 3-8
  photo?: boolean;      // hint: photorealistic vs illustration
}): Promise<string> {
  const res = await fetch("/api/generate-image", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({
      prompt: opts.prompt,
      size: opts.size ?? 1024,
      guidance: opts.guidance ?? 5,
      style: opts.photo ? "photo" : "illustration",
      transparent: true,
    }),
  });
  if (!res.ok) {
    const text = await res.text().catch(() => "");
    throw new Error(`Image generation failed: ${res.status} ${text}`);
  }
  const { dataUrl } = await res.json();
  return dataUrl as string; // "data:image/png;base64,...."
}

// Compose icon + text into a single transparent PNG
export async function composeIconWithText(opts: {
  iconUrl: string;        // PNG (transparent)
  name: string;           // brand name
  fontFamily: string;     // CSS font-family string
  color: string;          // text color (Primary)
  layout?: "left" | "top"; // icon left of text, or above
  gap?: number;           // px gap between icon and text
  maxWidth?: number;
  maxHeight?: number;
}): Promise<string> {
  const layout = opts.layout ?? "left";
  const gap = opts.gap ?? 24;
  const maxW = opts.maxWidth ?? 1200;
  const maxH = opts.maxHeight ?? 600;

  // Load icon image
  const img = await new Promise<HTMLImageElement>((res, rej) => {
    const im = new Image();
    im.onload = () => res(im);
    im.onerror = rej;
    im.src = opts.iconUrl;
  });

  // Create canvas
  const c = document.createElement("canvas");
  c.width = maxW;
  c.height = maxH;
  const g = c.getContext("2d");
  if (!g) throw new Error("Canvas 2D not available");
  g.clearRect(0, 0, maxW, maxH);

  // Compute icon box
  const iconMax = layout === "left" ? Math.min(maxH * 0.8, 360) : Math.min(maxW * 0.6, 360);
  const iconW = iconMax;
  const iconH = iconMax;

  // Decide font size to fit remaining space
  const textAreaW = layout === "left" ? maxW - iconW - gap * 2 : maxW * 0.98;
  let fontSize = layout === "left" ? Math.min(140, Math.max(48, Math.floor(maxH * 0.35))) : Math.min(120, Math.max(48, Math.floor(maxH * 0.35)));
  g.fillStyle = opts.color;
  g.textBaseline = "middle";
  g.textAlign = "left";

  // Fit-to-width loop
  for (let tries = 0; tries < 12; tries++) {
    g.font = `700 ${fontSize}px ${opts.fontFamily}`;
    const width = g.measureText(opts.name).width;
    if (width <= textAreaW) break;
    fontSize -= 6;
  }

  // Draw
  if (layout === "left") {
    const cx = gap + iconW / 2;
    const cy = maxH / 2;
    g.drawImage(img, cx - iconW / 2, cy - iconH / 2, iconW, iconH);
    g.font = `700 ${fontSize}px ${opts.fontFamily}`;
    g.fillText(opts.name, gap + iconW + gap, cy + (fontSize * 0.06)); // slight optical adjustment
  } else {
    const cx = maxW / 2;
    const top = gap;
    g.drawImage(img, cx - iconW / 2, top, iconW, iconH);
    g.font = `700 ${fontSize}px ${opts.fontFamily}`;
    g.textAlign = "center";
    g.textBaseline = "top";
    g.fillText(opts.name, cx, top + iconH + gap);
  }

  return c.toDataURL("image/png");
}

// Legacy function for backwards compatibility
export async function makePngLogo(
  opts: {
    name: string;            // brand name (for initial)
    bg: string;              // background color
    fg: string;              // foreground color
    prompt?: string;         // image description (not used in stub)
    badge?: boolean;         // draw circle badge
  } = { name: "A", bg: "#111827", fg: "#ffffff", prompt: "", badge: true }
): Promise<string> {
  const size = 512;
  const canvas = document.createElement("canvas");
  canvas.width = size;
  canvas.height = size;
  const ctx = canvas.getContext("2d");
  if (!ctx) throw new Error("Canvas 2D not available");

  // background
  ctx.fillStyle = opts.bg;
  ctx.fillRect(0, 0, size, size);

  // simple badge
  if (opts.badge) {
    ctx.beginPath();
    ctx.arc(size / 2, size / 2, size * 0.36, 0, Math.PI * 2);
    ctx.fillStyle = opts.fg + "ee".slice(0, 2); // slight alpha
    ctx.fill();
  }

  // initial / monogram
  const initial = (opts.name?.trim()?.[0] ?? "A").toUpperCase();
  ctx.fillStyle = opts.fg;
  ctx.font = "bold 220px Inter, Arial, Helvetica, sans-serif";
  ctx.textAlign = "center";
  ctx.textBaseline = "middle";
  ctx.fillText(initial, size / 2, size / 2);

  return canvas.toDataURL("image/png");
}