// src/utils/normalizeSvg.ts
// Ensures every incoming SVG string has a viewBox and scales correctly.

/**
 * SN's exact cleanSvg function - surgical approach to remove background rectangles
 */
export function cleanSvg(svgText: string): string {
  if (!svgText) return svgText;
  
  return svgText
    // remove any rect backgrounds (more aggressive than just white)
    .replace(/<rect[^>]+fill="[^"]*"[^>]*>/gi, "")
    // remove <background> blocks if they exist
    .replace(/<background[^>]*>.*?<\/background>/gi, "");
}

/**
 * Removes background fills from SVG to ensure transparency
 */
export function removeBackgroundFills(svgContent: string): string {
  if (!svgContent) return svgContent;

  let processed = svgContent;

  // First apply SN's simple cleaning function
  processed = cleanSvg(processed);

  // Remove background rectangles that cover the entire viewBox
  // Common patterns: <rect fill="white/color" width="100%" height="100%">
  processed = processed.replace(
    /<rect[^>]*(?:width\s*=\s*["'][^"']*100%[^"']*["']|height\s*=\s*["'][^"']*100%[^"']*["'])[^>]*fill\s*=\s*["'][^"']*["'][^>]*\/?>|<rect[^>]*fill\s*=\s*["'][^"']*["'][^>]*(?:width\s*=\s*["'][^"']*100%[^"']*["']|height\s*=\s*["'][^"']*100%[^"']*["'])[^>]*\/?>/gi,
    ''
  );

  // Remove background rectangles that match viewBox dimensions exactly
  const viewBoxMatch = processed.match(/viewBox\s*=\s*["']([^"']+)["']/i);
  if (viewBoxMatch) {
    const viewBox = viewBoxMatch[1].split(/\s+/);
    if (viewBox.length === 4) {
      const [x, y, w, h] = viewBox;
      // Remove rect elements that exactly match the viewBox dimensions
      const bgRectPattern = new RegExp(
        `<rect[^>]*(?:x\\s*=\\s*["']${x}["']|x\\s*=\\s*["']0["'])?[^>]*(?:y\\s*=\\s*["']${y}["']|y\\s*=\\s*["']0["'])?[^>]*(?:width\\s*=\\s*["']${w}["'])[^>]*(?:height\\s*=\\s*["']${h}["'])[^>]*\\/?\\s*>`,
        'gi'
      );
      processed = processed.replace(bgRectPattern, '');
    }
  }

  // Remove any remaining background rectangles with white/light fills that appear first
  processed = processed.replace(
    /<rect[^>]*fill\s*=\s*["'](?:#?(?:fff|ffffff|white|f{3,6}))["'][^>]*\/?>(?=(?:(?!<\/svg>).)*<(?:path|circle|ellipse|polygon|polyline))/gi,
    ''
  );

  // Remove any <g> tags that only contain background elements
  processed = processed.replace(
    /<g[^>]*>\s*<rect[^>]*fill\s*=\s*["'][^"']*["'][^>]*\/?\s*>\s*<\/g>/gi,
    ''
  );

  // Ensure the root SVG has no fill attribute that creates a background
  processed = processed.replace(
    /(<svg[^>]*)\s+fill\s*=\s*["'][^"']*["']/gi,
    '$1'
  );

  // Remove style attributes from the root SVG that set background
  processed = processed.replace(
    /(<svg[^>]*)\s+style\s*=\s*["'][^"']*background[^"']*["']/gi,
    '$1'
  );

  return processed;
}

export function normalizeSvg(svgRaw: string): string {
  if (!svgRaw) return svgRaw;

  // Quick sanity: trim and ensure it's an SVG string
  const trimmed = svgRaw.trim();
  if (!trimmed.startsWith("<svg")) return trimmed;

  // Parse attributes safely
  const openTagEnd = trimmed.indexOf(">");
  if (openTagEnd === -1) return trimmed;

  const openTag = trimmed.slice(0, openTagEnd + 1);
  const rest = trimmed.slice(openTagEnd + 1);

  // If there is already a viewBox, just ensure preserveAspectRatio
  const hasViewBox = /viewBox\s*=\s*['"][^'"]+['"]/i.test(openTag);

  // Extract width/height if present (could be like "512", "512px", "100%" etc.)
  const widthMatch = openTag.match(/\bwidth\s*=\s*['"]([^'"]+)['"]/i);
  const heightMatch = openTag.match(/\bheight\s*=\s*['"]([^'"]+)['"]/i);

  const widthVal = widthMatch?.[1] ?? "";
  const heightVal = heightMatch?.[1] ?? "";

  const numeric = (v: string) => {
    const n = parseFloat(v.replace(/px|pt|em|rem|%/g, ""));
    return Number.isFinite(n) && n > 0 ? n : undefined;
  };

  let w = numeric(widthVal);
  let h = numeric(heightVal);

  // Fallback square if neither width nor height can be parsed.
  if (!w && h) w = h;
  if (!h && w) h = w;
  if (!w && !h) {
    // Try to infer from common defaults; 512 is a sensible safety default
    w = 512; h = 512;
  }

  // Ensure preserveAspectRatio="xMidYMid meet"
  let newOpenTag = openTag;
  if (!/preserveAspectRatio=/i.test(newOpenTag)) {
    newOpenTag = newOpenTag.replace(
      /^<svg/i,
      `<svg preserveAspectRatio="xMidYMid meet"`
    );
  } else {
    newOpenTag = newOpenTag.replace(
      /preserveAspectRatio\s*=\s*['"][^'"]+['"]/i,
      `preserveAspectRatio="xMidYMid meet"`
    );
  }

  if (!hasViewBox) {
    newOpenTag = newOpenTag.replace(
      /^<svg/i,
      `<svg viewBox="0 0 ${w} ${h}"`
    );
  }

  // Remove explicit width/height so the SVG can flex to container
  newOpenTag = newOpenTag
    .replace(/\bwidth\s*=\s*['"][^'"]+['"]/gi, "")
    .replace(/\bheight\s*=\s*['"][^'"]+['"]/gi, "");

  return `${newOpenTag}${rest}`;
}

/**
 * Complete SVG processing: normalize structure and remove backgrounds
 */
export function processSvgForTransparency(svgRaw: string): string {
  const normalized = normalizeSvg(svgRaw);
  return removeBackgroundFills(normalized);
}