/**
 * Brand Kit binding for the app.
 * We try (in order):
 *  - React context (if your app provides one) — TODO: you can wire this later
 *  - localStorage key "ibrandbiz_brandkit"
 *  - safe defaults
 */

export type BrandKit = {
  name?: string;
  fonts?: {
    heading?: string;
    body?: string;
  };
  colors: {
    primary: string;
    accent?: string;
    neutral?: string;
    [key: string]: string | undefined;
  };
  logo?: {
    dataUrl?: string; // base64 data URL
  };
};

const DEFAULT_BRAND: BrandKit = {
  name: "IBrandBiz",
  fonts: { heading: "Inter", body: "Inter" },
  colors: {
    primary: "#05445e",
    accent: "#ff8800",
    neutral: "#d4f1f4",
  },
  logo: { dataUrl: "" },
};

export function useBrandKit(): BrandKit {
  // 1) If you later add a BrandKitContext, read it here.

  // 2) localStorage fallback
  try {
    const raw = localStorage.getItem("ibrandbiz_brandkit");
    if (raw) {
      const parsed = JSON.parse(raw);
      // Merge with defaults to avoid undefined fields
      return {
        name: parsed.name ?? DEFAULT_BRAND.name,
        fonts: {
          heading: parsed.fonts?.heading ?? DEFAULT_BRAND.fonts?.heading,
          body: parsed.fonts?.body ?? DEFAULT_BRAND.fonts?.body,
        },
        colors: {
          ...DEFAULT_BRAND.colors,
          ...(parsed.colors || {}),
        },
        logo: {
          dataUrl: parsed.logo?.dataUrl ?? DEFAULT_BRAND.logo?.dataUrl,
        },
      };
    }
  } catch {
    /* ignore */
  }

  // 3) Safe default
  return DEFAULT_BRAND;
}