import { create } from "zustand";
import { nanoid } from "nanoid";
import type { PlanState, PlanSection, SectionKind, SectionId, BusinessBrief } from "@shared/plan-types";
import { MIGRATION_PLACEHOLDERS } from "@shared/plan-section-templates";

const V2_STORAGE_KEY = "ibrandbiz.plan.v2";

// New 12-section default structure per SN's spec
const V2_DEFAULT_SECTIONS: PlanSection[] = [
  { id: nanoid(), kind: "executive-summary", title: "Executive Summary", content: "", isEnabled: true, sortIndex: 0, children: [] },
  { id: nanoid(), kind: "company-overview", title: "Company Overview", content: "", isEnabled: true, sortIndex: 1, children: [] },
  { id: nanoid(), kind: "products-services", title: "Products & Services", content: "", isEnabled: true, sortIndex: 2, children: [] },
  { id: nanoid(), kind: "market-analysis", title: "Market Analysis", content: "", isEnabled: true, sortIndex: 3, children: [] },
  { id: nanoid(), kind: "marketing-sales", title: "Marketing & Sales", content: "", isEnabled: true, sortIndex: 4, children: [] },
  { id: nanoid(), kind: "operations-plan", title: "Operations Plan", content: "", isEnabled: true, sortIndex: 5, children: [] },
  { id: nanoid(), kind: "org-management", title: "Organization & Management", content: "", isEnabled: true, sortIndex: 6, children: [] },
  { id: nanoid(), kind: "financial-plan", title: "Financial Plan", content: "", isEnabled: true, sortIndex: 7, children: [] },
  { id: nanoid(), kind: "milestones-roadmap", title: "Milestones & Roadmap", content: "", isEnabled: true, sortIndex: 8, children: [] },
  { id: nanoid(), kind: "future-expansion", title: "Future Expansion & Ecosystem Vision", content: "", isEnabled: true, sortIndex: 9, children: [] },
  { id: nanoid(), kind: "partnerships-note", title: "Partnerships Note", content: "", isEnabled: true, sortIndex: 10, children: [] },
  { id: nanoid(), kind: "exit-strategy", title: "Exit Strategy", content: "", isOptional: true, isEnabled: false, sortIndex: 11, children: [] },
];

function migratePlan(plan: PlanState): PlanState {
  // Check if this specific plan needs migration by inspecting its sections
  const existingKinds = new Set(plan.sections.map(s => s.kind));
  
  // If plan already has all 4 new sections, no migration needed
  const hasAllNewSections = 
    existingKinds.has("milestones-roadmap") &&
    existingKinds.has("future-expansion") &&
    existingKinds.has("partnerships-note") &&
    existingKinds.has("exit-strategy");
  
  if (hasAllNewSections) return plan;

  const newSections: PlanSection[] = [];

  // Add missing sections with placeholder content
  if (!existingKinds.has("milestones-roadmap")) {
    newSections.push({
      id: nanoid(),
      kind: "milestones-roadmap",
      title: "Milestones & Roadmap",
      content: MIGRATION_PLACEHOLDERS["milestones-roadmap"],
      isEnabled: true,
      sortIndex: 8,
      children: []
    });
  }

  if (!existingKinds.has("future-expansion")) {
    newSections.push({
      id: nanoid(),
      kind: "future-expansion",
      title: "Future Expansion & Ecosystem Vision",
      content: MIGRATION_PLACEHOLDERS["future-expansion"],
      isEnabled: true,
      sortIndex: 9,
      children: []
    });
  }

  if (!existingKinds.has("partnerships-note")) {
    newSections.push({
      id: nanoid(),
      kind: "partnerships-note",
      title: "Partnerships Note",
      content: MIGRATION_PLACEHOLDERS["partnerships-note"],
      isEnabled: true,
      sortIndex: 10,
      children: []
    });
  }

  if (!existingKinds.has("exit-strategy")) {
    newSections.push({
      id: nanoid(),
      kind: "exit-strategy",
      title: "Exit Strategy",
      content: MIGRATION_PLACEHOLDERS["exit-strategy"],
      isOptional: true,
      isEnabled: false,
      sortIndex: 11,
      children: []
    });
  }

  // Update existing sections to have new fields if missing
  const updatedSections = plan.sections.map((sec, index) => ({
    ...sec,
    isEnabled: sec.isEnabled !== false,
    sortIndex: sec.sortIndex ?? index,
  }));

  return {
    ...plan,
    sections: [...updatedSections, ...newSections]
  };
}

function v2Load(): PlanState {
  try {
    const raw = localStorage.getItem(V2_STORAGE_KEY);
    if (raw) {
      const loaded = JSON.parse(raw) as PlanState;
      return migratePlan(loaded);
    }
  } catch {}
  return { planId: nanoid(), title: "Untitled Business Plan", sections: V2_DEFAULT_SECTIONS };
}

type V2Store = PlanState & {
  addSection(kind?: SectionKind, title?: string): SectionId;
  addChildSection(parentId: SectionId, title?: string): SectionId | null;
  removeSection(id: SectionId): void;
  removeChildSection(parentId: SectionId, childId: SectionId): void;
  reorderSections(idsInOrder: SectionId[]): void;
  reorderChildSections(parentId: SectionId, idsInOrder: SectionId[]): void;
  updateSection(id: SectionId, patch: Partial<PlanSection>): void;
  updateChildSection(parentId: SectionId, childId: SectionId, patch: Partial<PlanSection>): void;
  toggleSectionEnabled(id: SectionId): void;
  duplicateSection(id: SectionId): SectionId | null;
  resetToDefault(): void;
  setPlanTitle(title: string, isManualEdit?: boolean): void;
  updateBusinessBrief(brief: Partial<BusinessBrief>): void;
};

export const usePlanStoreV2 = create<V2Store>((set, get) => ({
  ...v2Load(),
  addSection: (kind = "custom", title = "New Section") => {
    const sec: PlanSection = { id: nanoid(), kind, title, content: "", children: [] };
    set((s) => ({ sections: [...s.sections, sec] }));
    v2Persist();
    return sec.id;
  },
  addChildSection: (parentId, title = "New Subsection") => {
    set((s) => ({
      sections: s.sections.map((sec) =>
        sec.id === parentId
          ? { ...sec, children: [...(sec.children || []), { id: nanoid(), kind: "custom", title, content: "" }] }
          : sec
      ),
    }));
    v2Persist();
    const p = get().sections.find((x) => x.id === parentId);
    const last = p?.children?.[p.children.length - 1];
    return last?.id || null;
  },
  removeSection: (id) => { 
    set((s) => ({ sections: s.sections.filter((x) => x.id !== id) })); 
    v2Persist(); 
  },
  removeChildSection: (parentId, childId) => {
    set((s) => ({
      sections: s.sections.map((sec) => 
        sec.id === parentId 
          ? { ...sec, children: (sec.children||[]).filter((c)=>c.id!==childId) } 
          : sec
      )
    }));
    v2Persist();
  },
  reorderSections: (ids) => { 
    set((s) => ({ 
      sections: ids.map((id) => s.sections.find((x)=>x.id===id)!).filter(Boolean) 
    })); 
    v2Persist(); 
  },
  reorderChildSections: (parentId, ids) => {
    set((s) => ({
      sections: s.sections.map((sec) => {
        if (sec.id !== parentId) return sec;
        const current = sec.children || [];
        const out = ids.map((id) => current.find((c)=>c.id===id)!).filter(Boolean);
        return { ...sec, children: out };
      })
    }));
    v2Persist();
  },
  updateSection: (id, patch) => { 
    set((s)=>({ 
      sections: s.sections.map((sec)=>sec.id===id?{...sec, ...patch}:sec) 
    })); 
    v2Persist(); 
  },
  updateChildSection: (parentId, childId, patch) => {
    set((s)=>({ 
      sections: s.sections.map((sec)=> 
        sec.id!==parentId? sec : { 
          ...sec, 
          children: (sec.children||[]).map((c)=> c.id===childId?{...c, ...patch}:c) 
        }) 
    }));
    v2Persist();
  },
  toggleSectionEnabled: (id) => {
    set((s) => ({
      sections: s.sections.map((sec) =>
        sec.id === id ? { ...sec, isEnabled: !sec.isEnabled, lastUpdated: new Date().toISOString() } : sec
      )
    }));
    v2Persist();
  },
  duplicateSection: (id) => {
    const src = get().sections.find((x)=>x.id===id); 
    if (!src) return null;
    const dupe: PlanSection = { ...src, id: nanoid(), title: `${src.title} (Copy)` };
    set((s)=>({ sections: [...s.sections, dupe] })); 
    v2Persist(); 
    return dupe.id;
  },
  resetToDefault: () => {
    const state = { planId: nanoid(), title: "Untitled Business Plan", sections: V2_DEFAULT_SECTIONS };
    set(state); 
    try { localStorage.setItem(V2_STORAGE_KEY, JSON.stringify(state)); } catch {}
  },
  setPlanTitle: (title, isManualEdit = true) => {
    set((s) => ({ 
      title, 
      planTitleManuallyEdited: isManualEdit ? true : s.planTitleManuallyEdited 
    }));
    v2Persist();
  },
  updateBusinessBrief: (brief) => {
    set((s) => ({
      businessBrief: { ...s.businessBrief, ...brief }
    }));
    v2Persist();
  },
}));

function v2Persist(){
  try {
    const { planId, title, planTitleManuallyEdited, sections, businessBrief } = usePlanStoreV2.getState();
    localStorage.setItem(V2_STORAGE_KEY, JSON.stringify({ planId, title, planTitleManuallyEdited, sections, businessBrief }));
  } catch {}
}