import { create } from "zustand";
import { nanoid } from "nanoid";
import type { PlanState, PlanSection, SectionKind, SectionId } from "@shared/plan-types";

const DEFAULT_SECTIONS: PlanSection[] = [
  { id: nanoid(), kind: "executive-summary", title: "Executive Summary", content: "" },
  { id: nanoid(), kind: "company-overview", title: "Company Overview", content: "" },
  { id: nanoid(), kind: "products-services", title: "Products & Services", content: "" },
  { id: nanoid(), kind: "market-analysis", title: "Market Analysis", content: "" },
  { id: nanoid(), kind: "marketing-sales", title: "Marketing & Sales", content: "" },
  { id: nanoid(), kind: "operations", title: "Operations Plan", content: "" },
  { id: nanoid(), kind: "org-management", title: "Organization & Management", content: "" },
  { id: nanoid(), kind: "financials", title: "Financial Plan", content: "" },
];

type Store = PlanState & {
  addSection(kind?: SectionKind, title?: string): SectionId;
  removeSection(id: SectionId): void;
  reorderSections(idsInOrder: SectionId[]): void;
  updateSection(id: SectionId, patch: Partial<PlanSection>): void;
  duplicateSection(id: SectionId): SectionId | null;
  resetToDefault(): void;
  setPlanTitle(title: string): void;
};

export const usePlanStore = create<Store>((set, get) => ({
  planId: nanoid(),
  title: "Untitled Business Plan",
  sections: DEFAULT_SECTIONS,

  addSection: (kind = "custom", title = "New Section") => {
    const section: PlanSection = { id: nanoid(), kind, title, content: "" };
    set((s) => ({ sections: [...s.sections, section] }));
    return section.id;
  },

  removeSection: (id) => 
    set((s) => ({ sections: s.sections.filter((sec) => sec.id !== id) })),

  reorderSections: (ids) =>
    set((s) => ({
      sections: ids
        .map((id) => s.sections.find((x) => x.id === id)!)
        .filter(Boolean),
    })),

  updateSection: (id, patch) =>
    set((s) => ({ 
      sections: s.sections.map((sec) => (sec.id === id ? { ...sec, ...patch } : sec)) 
    })),

  duplicateSection: (id) => {
    const src = get().sections.find((x) => x.id === id);
    if (!src) return null;
    const dupe: PlanSection = { ...src, id: nanoid(), title: `${src.title} (Copy)` };
    set((s) => ({ sections: [...s.sections, dupe] }));
    return dupe.id;
  },

  resetToDefault: () => 
    set({ sections: DEFAULT_SECTIONS }),

  setPlanTitle: (title) =>
    set({ title }),
}));