import { create } from "zustand";
import { nanoid } from "nanoid";
import type { FinanceInputsV2, FinanceOutputsV2, ExpenseRow, MonthlyRowV2 } from "@shared/finance-types";

const KEY_V2 = "ibrandbiz.finance.v2";

export type FinanceStoreV2 = {
  inputs: FinanceInputsV2;
  outputs: FinanceOutputsV2 | null;
  setInputs: (patch: Partial<FinanceInputsV2>) => void;
  setExpenses: (expenses: ExpenseRow[]) => void;
  addExpense: (name?: string, amount?: number) => void;
  updateExpense: (id: string, patch: Partial<ExpenseRow>) => void;
  deleteExpense: (id: string) => void;
  recompute: () => void;
  reset: () => void;
};

const DEFAULT_EXPENSES: ExpenseRow[] = [
  { id: nanoid(), name: "Rent & Utilities", amount: 1200 },
  { id: nanoid(), name: "Software & Tools", amount: 800 },
  { id: nanoid(), name: "Marketing", amount: 500 },
  { id: nanoid(), name: "Misc", amount: 500 },
];

const DEFAULT_INPUTS_V2: FinanceInputsV2 = {
  currency: "USD",
  startMonth: "2025-10",
  pricePerUnit: 49,
  unitsPerMonth: [10,12,15,18,22,26,30,35,40,46,52,60],
  cogsPercent: 30,
  fixedMonthlyExpenses: 3000, // auto-calculated from expenses
  variableExpensePerUnit: 0,
  startingCash: 5000,
  expenses: DEFAULT_EXPENSES,
};

function loadV2(): FinanceInputsV2 {
  try { 
    const raw = localStorage.getItem(KEY_V2); 
    if (raw) {
      const parsed = JSON.parse(raw);
      // Ensure expenses exist and have valid structure
      if (!parsed.expenses || !Array.isArray(parsed.expenses)) {
        parsed.expenses = DEFAULT_EXPENSES;
      }
      // Auto-calculate fixedMonthlyExpenses from expenses
      parsed.fixedMonthlyExpenses = parsed.expenses.reduce((sum: number, exp: ExpenseRow) => sum + (exp.amount || 0), 0);
      return parsed;
    }
  } catch {}
  return DEFAULT_INPUTS_V2;
}

function persistV2(inputs: FinanceInputsV2) {
  try { 
    localStorage.setItem(KEY_V2, JSON.stringify(inputs)); 
  } catch {}
}

const MONTH_NAMES = ["Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"];

function formatMonthLabel(ym: string, i: number) {
  const [y0, m0] = ym.split("-").map(Number);
  const base = new Date(y0, (m0 - 1) + i, 1);
  return `${MONTH_NAMES[base.getMonth()]} ${base.getFullYear()}`;
}

function computePAndLV2(input: FinanceInputsV2): FinanceOutputsV2 {
  const rows: MonthlyRowV2[] = [];
  let cumulativeProfit = 0;
  let currentCash = input.startingCash || 0;
  
  const price = Number(input.pricePerUnit) || 0;
  const cogsPct = Math.min(Math.max(input.cogsPercent, 0), 100) / 100;
  const varPerUnit = Number(input.variableExpensePerUnit || 0);
  const opex = Number(input.fixedMonthlyExpenses) || 0;

  let lowestCash = currentCash;
  let lowestCashMonth = 0;

  for (let i = 0; i < 12; i++) {
    const units = Number(input.unitsPerMonth[i] || 0);
    const revenue = units * price;
    const cogs = revenue * cogsPct + units * varPerUnit;
    const grossProfit = revenue - cogs;
    const profit = grossProfit - opex;
    cumulativeProfit += profit;
    currentCash += profit;

    if (currentCash < lowestCash) {
      lowestCash = currentCash;
      lowestCashMonth = i;
    }

    rows.push({
      month: formatMonthLabel(input.startMonth, i),
      revenue, 
      cogs, 
      grossProfit, 
      opex, 
      profit, 
      cumulativeProfit,
      endingCash: currentCash,
    });
  }

  const totalRevenue = rows.reduce((s, r) => s + r.revenue, 0);
  const totalProfit = rows.reduce((s, r) => s + r.profit, 0);

  // breakeven = first month where cumulativeProfit >= 0
  let breakevenMonthIndex: number | null = null;
  for (let i = 0; i < rows.length; i++) {
    if (rows[i].cumulativeProfit >= 0) { 
      breakevenMonthIndex = i; 
      break; 
    }
  }

  let runwayMonths: number | null = null;
  if (input.startingCash != null) {
    let cash = input.startingCash;
    let m = 0;
    while (m < 60) { // cap at 5 years
      const idx = Math.min(m, rows.length - 1);
      cash += rows[idx].profit;
      if (cash <= 0) { 
        runwayMonths = m + 1; 
        break; 
      }
      m++;
    }
    if (runwayMonths == null) runwayMonths = 60; // survival beyond horizon
  }

  return { 
    rows, 
    totalRevenue, 
    totalProfit, 
    breakevenMonthIndex, 
    runwayMonths,
    cashFlowAnalysis: {
      startingCash: input.startingCash || 0,
      lowestCash,
      lowestCashMonth,
      endingCash: currentCash,
    }
  };
}

export const useFinanceStoreV2 = create<FinanceStoreV2>((set, get) => ({
  inputs: loadV2(),
  outputs: null,
  
  setInputs: (patch) => {
    const newInputs = { ...get().inputs, ...patch };
    set({ inputs: newInputs });
    persistV2(newInputs);
  },

  setExpenses: (expenses) => {
    const fixedMonthlyExpenses = expenses.reduce((sum, exp) => sum + (exp.amount || 0), 0);
    const newInputs = { 
      ...get().inputs, 
      expenses, 
      fixedMonthlyExpenses 
    };
    set({ inputs: newInputs });
    persistV2(newInputs);
  },

  addExpense: (name = "New Expense", amount = 0) => {
    const newExpense: ExpenseRow = { id: nanoid(), name, amount };
    const newExpenses = [...get().inputs.expenses, newExpense];
    get().setExpenses(newExpenses);
  },

  updateExpense: (id, patch) => {
    const newExpenses = get().inputs.expenses.map(exp => 
      exp.id === id ? { ...exp, ...patch } : exp
    );
    get().setExpenses(newExpenses);
  },

  deleteExpense: (id) => {
    const newExpenses = get().inputs.expenses.filter(exp => exp.id !== id);
    get().setExpenses(newExpenses);
  },

  recompute: () => {
    const outputs = computePAndLV2(get().inputs);
    set({ outputs });
  },

  reset: () => {
    set({ inputs: DEFAULT_INPUTS_V2, outputs: null });
    persistV2(DEFAULT_INPUTS_V2);
  },
}));

// Auto-compute on mount
useFinanceStoreV2.getState().recompute();