import { create } from "zustand";
import type { FinanceInputs, FinanceOutputs } from "@shared/finance-types";
import { computePAndL } from "@/lib/financeCalc";

const KEY = "ibrandbiz.finance.v1";

export type FinanceStore = {
  inputs: FinanceInputs;
  outputs: FinanceOutputs | null;
  setInputs: (patch: Partial<FinanceInputs>) => void;
  recompute: () => void;
  reset: () => void;
};

const DEFAULT_INPUTS: FinanceInputs = {
  currency: "USD",
  startMonth: "2025-10",
  pricePerUnit: 49,
  unitsPerMonth: [10,12,15,18,22,26,30,35,40,46,52,60],
  cogsPercent: 30,
  fixedMonthlyExpenses: 3000,
  variableExpensePerUnit: 0,
  startingCash: 5000,
};

function load(): FinanceInputs {
  try { const raw = localStorage.getItem(KEY); if (raw) return JSON.parse(raw); } catch {}
  return DEFAULT_INPUTS;
}

export const useFinanceStore = create<FinanceStore>((set, get) => ({
  inputs: load(),
  outputs: null,
  setInputs: (patch) => {
    set((s) => ({ inputs: { ...s.inputs, ...patch } }));
    persist(get().inputs);
  },
  recompute: () => {
    const out = computePAndL(get().inputs);
    set({ outputs: out });
  },
  reset: () => {
    set({ inputs: DEFAULT_INPUTS, outputs: null });
    persist(DEFAULT_INPUTS);
  },
}));

function persist(inputs: FinanceInputs) {
  try { localStorage.setItem(KEY, JSON.stringify(inputs)); } catch {}
}

// Auto-compute on mount
useFinanceStore.getState().recompute();