import { create } from "zustand";
import { persist, createJSONStorage } from "zustand/middleware";
import type { BrandKit } from "@shared/brand-types";

const LS_KEY = "ibrandbiz-brand-kits-v2";

function uid(): string {
  return Math.random().toString(36).substr(2, 9);
}

const DEFAULT_KIT: BrandKit = {
  id: "default",
  name: "IBrandBiz Default",
  primary: "#6D28D9",
  secondary: "#06B6D4", 
  accent: "#F59E0B",
  logoUrl: "",
  fontHead: "Inter, ui-sans-serif",
  fontBody: "Inter, ui-sans-serif",
};

export interface BrandKitStore {
  kits: BrandKit[];
  current: BrandKit;
  loadFromStorage: () => void;
  createKit: (kit: Omit<BrandKit, "id">) => string;
  updateKit: (id: string, updates: Partial<BrandKit>) => void;
  deleteKit: (id: string) => void;
  setDefault: (id: string) => void;
  getCurrentKit: () => BrandKit;
}

export const useBrandKitStore = create<BrandKitStore>()(
  persist(
    (set, get) => ({
      kits: [DEFAULT_KIT],
      current: DEFAULT_KIT,
      
      loadFromStorage: () => {
        try {
          const stored = localStorage.getItem(LS_KEY);
          if (stored) {
            const data = JSON.parse(stored);
            set({ 
              kits: data.kits || [DEFAULT_KIT], 
              current: data.current || DEFAULT_KIT 
            });
          }
        } catch (e) {
          console.warn("Failed to load brand kits from storage:", e);
        }
      },
      
      createKit: (kit) => {
        const id = uid();
        const newKit = { ...kit, id };
        set((state) => ({
          kits: [...state.kits, newKit],
          current: newKit
        }));
        return id;
      },
      
      updateKit: (id, updates) => {
        set((state) => {
          const updatedKits = state.kits.map(kit => 
            kit.id === id ? { ...kit, ...updates } : kit
          );
          const updatedCurrent = state.current.id === id 
            ? { ...state.current, ...updates }
            : state.current;
          return { kits: updatedKits, current: updatedCurrent };
        });
      },
      
      deleteKit: (id) => {
        if (id === "default") return; // Can't delete default
        set((state) => {
          const filteredKits = state.kits.filter(kit => kit.id !== id);
          const newCurrent = state.current.id === id 
            ? filteredKits[0] || DEFAULT_KIT
            : state.current;
          return { kits: filteredKits, current: newCurrent };
        });
      },
      
      setDefault: (id) => {
        const kit = get().kits.find(k => k.id === id);
        if (kit) {
          set({ current: kit });
        }
      },
      
      getCurrentKit: () => get().current,
    }),
    {
      name: LS_KEY,
      storage: createJSONStorage(() => localStorage),
    }
  )
);