import { useEffect, useMemo, useState } from "react";
import {
  ensureFirebase,
  isFirebaseConfigured,
  doc,
  onSnapshot,
  setDoc,
  updateDoc,
  arrayUnion,
  arrayRemove,
} from "../lib/firebase";

// -------- Local fallback --------
const LS_KEY = "ibrandbiz_icon_favorites";
function lsLoad(): string[] {
  try {
    const raw = localStorage.getItem(LS_KEY);
    const arr = raw ? JSON.parse(raw) : [];
    return Array.isArray(arr) ? arr : [];
  } catch {
    return [];
  }
}
function lsSave(ids: string[]) {
  try {
    localStorage.setItem(LS_KEY, JSON.stringify(ids));
  } catch {}
}

// -------- Hook API (cloud-first, local fallback) --------
export function useFavorites() {
  const [favorites, setFavorites] = useState<string[]>(lsLoad());
  const [ready, setReady] = useState<boolean>(!isFirebaseConfigured()); // if no Firebase, we're already "ready"
  const [cloud, setCloud] = useState<{ uid: string } | null>(null);

  useEffect(() => {
    const fb = ensureFirebase();
    if (!fb) {
      // no firebase config -> local only
      setReady(true);
      return;
    }

    let unsubscribe: (() => void) | null = null;

    (async () => {
      try {
        const { db, authReady } = fb;
        const user = await authReady;
        if (!user) {
          setReady(true);
          return;
        }
        setCloud({ uid: user.uid });

        const ref = doc(db, "users", user.uid, "favorites", "icons");
        // ensure doc exists
        await setDoc(ref, { ids: lsLoad() }, { merge: true });

        // subscribe
        unsubscribe = onSnapshot(ref, (snap) => {
          const data = snap.data() as any;
          const ids: string[] = Array.isArray(data?.ids) ? data.ids : [];
          setFavorites(ids);
          lsSave(ids); // keep local cache in sync
          setReady(true);
        });
      } catch (error) {
        console.error('Firebase favorites setup failed, falling back to local:', error);
        setReady(true);
      }
    })();

    return () => {
      if (unsubscribe) {
        unsubscribe();
      }
    };
  }, []);

  const api = useMemo(() => {
    async function toggleFavorite(id: string) {
      // cloud path
      if (cloud && isFirebaseConfigured()) {
        const fb = ensureFirebase();
        if (fb) {
          try {
            const ref = doc(fb.db, "users", cloud.uid, "favorites", "icons");
            const has = favorites.includes(id);
            await updateDoc(ref, { ids: has ? arrayRemove(id) : arrayUnion(id) });
            return;
          } catch (error) {
            console.error('Failed to update cloud favorites, falling back to local:', error);
            // Fall through to local update
          }
        }
      }
      // local fallback
      const cur = lsLoad();
      const next = cur.includes(id) ? cur.filter((x) => x !== id) : [...cur, id];
      lsSave(next);
      // optimistic UI update
      setFavorites(next);
    }

    function isFavorite(id: string) {
      return favorites.includes(id);
    }

    return { favorites, ready, toggleFavorite, isFavorite, cloudEnabled: !!cloud };
  }, [favorites, ready, cloud]);

  return api;
}