import type { WebsiteTemplate } from '@shared/schema';
import DOMPurify from 'dompurify';

interface TemplateCustomization {
  colors: {
    primary: string;
    secondary: string;
    accent: string;
    background: string;
    text: string;
  };
  fonts: {
    heading: string;
    body: string;
    sizes: {
      h1: string;
      h2: string;
      h3: string;
      body: string;
    };
  };
  layout: {
    spacing: string;
    maxWidth: string;
    gridColumns: number;
  };
  content: {
    sections: Array<{
      id: string;
      type: string;
      content: any;
      order: number;
    }>;
  };
}

interface SectionData {
  id: string;
  type: string;
  content: any;
  order: number;
  isVisible?: boolean;
}

interface TemplateStyles {
  cssVariables: Record<string, string>;
  baseStyles: string;
  sectionStyles: string;
  responsiveStyles: string;
}

interface RenderOptions {
  deviceMode?: 'desktop' | 'tablet' | 'mobile';
  includeInteractivity?: boolean;
  optimizeForExport?: boolean;
}

export class TemplateRenderer {
  private customization: TemplateCustomization;
  private template: WebsiteTemplate;

  constructor(template: WebsiteTemplate, customization: TemplateCustomization) {
    this.template = template;
    this.customization = customization;
  }

  /**
   * Generate CSS variables from customization data
   */
  private generateCSSVariables(): Record<string, string> {
    const spacing = this.getSpacingValue(this.customization.layout.spacing);
    
    return {
      '--primary': this.customization.colors.primary,
      '--primary-rgb': this.hexToRgb(this.customization.colors.primary),
      '--secondary': this.customization.colors.secondary,
      '--secondary-rgb': this.hexToRgb(this.customization.colors.secondary),
      '--accent': this.customization.colors.accent,
      '--accent-rgb': this.hexToRgb(this.customization.colors.accent),
      '--background': this.customization.colors.background,
      '--background-rgb': this.hexToRgb(this.customization.colors.background),
      '--text': this.customization.colors.text,
      '--text-rgb': this.hexToRgb(this.customization.colors.text),
      '--font-heading': `"${this.customization.fonts.heading}", system-ui, -apple-system, sans-serif`,
      '--font-body': `"${this.customization.fonts.body}", system-ui, -apple-system, sans-serif`,
      '--font-size-h1': this.customization.fonts.sizes.h1,
      '--font-size-h2': this.customization.fonts.sizes.h2,
      '--font-size-h3': this.customization.fonts.sizes.h3,
      '--font-size-body': this.customization.fonts.sizes.body,
      '--layout-max-width': this.customization.layout.maxWidth,
      '--layout-spacing': spacing,
      '--layout-spacing-sm': `calc(${spacing} * 0.5)`,
      '--layout-spacing-lg': `calc(${spacing} * 1.5)`,
      '--layout-grid-cols': this.customization.layout.gridColumns.toString(),
      '--border-radius': '0.5rem',
      '--border-radius-lg': '0.75rem',
      '--shadow-sm': '0 1px 2px 0 rgba(0, 0, 0, 0.05)',
      '--shadow': '0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06)',
      '--shadow-lg': '0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05)',
    };
  }

  /**
   * Generate base CSS styles
   */
  private generateBaseStyles(): string {
    return `
      * {
        margin: 0;
        padding: 0;
        box-sizing: border-box;
      }
      
      html {
        scroll-behavior: smooth;
      }
      
      body {
        font-family: var(--font-body);
        font-size: var(--font-size-body);
        line-height: 1.6;
        color: var(--text);
        background-color: var(--background);
        -webkit-font-smoothing: antialiased;
        -moz-osx-font-smoothing: grayscale;
      }
      
      h1, h2, h3, h4, h5, h6 {
        font-family: var(--font-heading);
        font-weight: 700;
        line-height: 1.2;
        margin-bottom: 1rem;
        color: var(--text);
      }
      
      h1 { font-size: var(--font-size-h1); }
      h2 { font-size: var(--font-size-h2); }
      h3 { font-size: var(--font-size-h3); }
      
      p {
        margin-bottom: 1rem;
        font-size: var(--font-size-body);
      }
      
      a {
        color: var(--primary);
        text-decoration: none;
        transition: color 0.3s ease;
      }
      
      a:hover {
        color: var(--secondary);
      }
      
      img {
        max-width: 100%;
        height: auto;
      }
      
      .container {
        max-width: var(--layout-max-width);
        margin: 0 auto;
        padding: 0 var(--layout-spacing);
      }
      
      .section {
        padding: calc(var(--layout-spacing) * 2) 0;
      }
      
      .section:nth-child(even) {
        background-color: rgba(var(--primary-rgb), 0.03);
      }
      
      .btn {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        padding: 0.75rem 1.5rem;
        font-size: var(--font-size-body);
        font-weight: 600;
        text-decoration: none;
        border: none;
        border-radius: var(--border-radius);
        cursor: pointer;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        user-select: none;
      }
      
      .btn-primary {
        background-color: var(--primary);
        color: white;
      }
      
      .btn-primary:hover {
        background-color: var(--secondary);
        transform: translateY(-1px);
        box-shadow: var(--shadow-lg);
      }
      
      .btn-secondary {
        background-color: transparent;
        color: var(--primary);
        border: 2px solid var(--primary);
      }
      
      .btn-secondary:hover {
        background-color: var(--primary);
        color: white;
        transform: translateY(-1px);
      }
      
      .btn-accent {
        background-color: var(--accent);
        color: white;
      }
      
      .btn-accent:hover {
        background-color: rgba(var(--accent-rgb), 0.9);
        transform: translateY(-1px);
        box-shadow: var(--shadow-lg);
      }
      
      .grid {
        display: grid;
        gap: var(--layout-spacing);
      }
      
      .grid-cols-1 { grid-template-columns: repeat(1, minmax(0, 1fr)); }
      .grid-cols-2 { grid-template-columns: repeat(2, minmax(0, 1fr)); }
      .grid-cols-3 { grid-template-columns: repeat(3, minmax(0, 1fr)); }
      .grid-cols-4 { grid-template-columns: repeat(4, minmax(0, 1fr)); }
      
      .text-center { text-align: center; }
      .text-left { text-align: left; }
      .text-right { text-align: right; }
      
      .mb-2 { margin-bottom: 0.5rem; }
      .mb-4 { margin-bottom: 1rem; }
      .mb-6 { margin-bottom: 1.5rem; }
      .mb-8 { margin-bottom: 2rem; }
      
      .mt-2 { margin-top: 0.5rem; }
      .mt-4 { margin-top: 1rem; }
      .mt-6 { margin-top: 1.5rem; }
      .mt-8 { margin-top: 2rem; }
    `;
  }

  /**
   * Generate section-specific styles
   */
  private generateSectionStyles(): string {
    return `
      .hero-section {
        background: linear-gradient(135deg, var(--primary), var(--secondary));
        color: white;
        text-align: center;
        padding: 6rem 0;
        position: relative;
        overflow: hidden;
      }
      
      .hero-section::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.1);
        z-index: 1;
      }
      
      .hero-section .container {
        position: relative;
        z-index: 2;
      }
      
      .hero-section h1,
      .hero-section h2,
      .hero-section h3 {
        color: white;
      }
      
      .hero-title {
        font-size: var(--font-size-h1);
        margin-bottom: 1rem;
        text-shadow: 0 2px 4px rgba(0, 0, 0, 0.3);
      }
      
      .hero-subtitle {
        font-size: calc(var(--font-size-body) * 1.25);
        margin-bottom: 2rem;
        opacity: 0.9;
        max-width: 600px;
        margin-left: auto;
        margin-right: auto;
      }
      
      .hero-actions {
        display: flex;
        gap: 1rem;
        justify-content: center;
        flex-wrap: wrap;
        margin-top: 2rem;
      }
      
      .features-section .container {
        text-align: center;
      }
      
      .features-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
        gap: 2rem;
        margin-top: 3rem;
      }
      
      .feature-item {
        padding: 2rem;
        background-color: var(--background);
        border-radius: var(--border-radius-lg);
        box-shadow: var(--shadow);
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        border: 1px solid rgba(var(--primary-rgb), 0.1);
      }
      
      .feature-item:hover {
        transform: translateY(-4px);
        box-shadow: var(--shadow-lg);
        border-color: rgba(var(--primary-rgb), 0.2);
      }
      
      .feature-icon {
        font-size: 2.5rem;
        margin-bottom: 1rem;
        color: var(--primary);
      }
      
      .feature-item h3 {
        color: var(--primary);
        margin-bottom: 0.75rem;
      }
      
      .about-section,
      .services-section {
        padding: 4rem 0;
      }
      
      .about-section .container,
      .services-section .container {
        max-width: 800px;
        text-align: center;
      }
      
      .gallery-section {
        padding: 4rem 0;
        background-color: rgba(var(--secondary-rgb), 0.05);
      }
      
      .gallery-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 1.5rem;
        margin-top: 2rem;
      }
      
      .gallery-item {
        aspect-ratio: 16/10;
        background-color: rgba(var(--primary-rgb), 0.1);
        border-radius: var(--border-radius);
        display: flex;
        align-items: center;
        justify-content: center;
        color: var(--text);
        font-weight: 600;
        transition: transform 0.3s ease;
      }
      
      .gallery-item:hover {
        transform: scale(1.02);
      }
      
      .contact-section {
        padding: 4rem 0;
        background-color: rgba(var(--accent-rgb), 0.05);
      }
      
      .contact-form {
        max-width: 600px;
        margin: 2rem auto 0;
        display: grid;
        gap: 1.5rem;
      }
      
      .form-group {
        display: flex;
        flex-direction: column;
        gap: 0.5rem;
      }
      
      .form-label {
        font-weight: 600;
        color: var(--text);
        font-size: 0.9rem;
      }
      
      .form-input,
      .form-textarea {
        padding: 0.75rem 1rem;
        border: 2px solid rgba(var(--primary-rgb), 0.2);
        border-radius: var(--border-radius);
        font-size: var(--font-size-body);
        font-family: var(--font-body);
        transition: all 0.3s ease;
        background-color: var(--background);
        color: var(--text);
      }
      
      .form-input:focus,
      .form-textarea:focus {
        outline: none;
        border-color: var(--primary);
        box-shadow: 0 0 0 3px rgba(var(--primary-rgb), 0.1);
      }
      
      .form-textarea {
        min-height: 120px;
        resize: vertical;
      }
      
      .footer-section {
        background-color: var(--text);
        color: var(--background);
        padding: 3rem 0 2rem;
        text-align: center;
      }
      
      .footer-section a {
        color: var(--background);
      }
      
      .footer-section a:hover {
        color: rgba(var(--background-rgb), 0.8);
      }
      
      .section-title {
        font-size: var(--font-size-h2);
        color: var(--primary);
        text-align: center;
        margin-bottom: 1rem;
      }
      
      .section-subtitle {
        font-size: calc(var(--font-size-body) * 1.1);
        color: rgba(var(--text-rgb), 0.8);
        text-align: center;
        margin-bottom: 3rem;
        max-width: 600px;
        margin-left: auto;
        margin-right: auto;
      }
    `;
  }

  /**
   * Generate responsive styles
   */
  private generateResponsiveStyles(): string {
    return `
      @media (max-width: 1024px) {
        .container {
          padding: 0 calc(var(--layout-spacing) * 0.75);
        }
        
        .features-grid {
          grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
          gap: 1.5rem;
        }
      }
      
      @media (max-width: 768px) {
        .hero-section {
          padding: 4rem 0;
        }
        
        .hero-title {
          font-size: calc(var(--font-size-h1) * 0.8);
        }
        
        .hero-actions {
          flex-direction: column;
          align-items: center;
          gap: 0.75rem;
        }
        
        .hero-actions .btn {
          width: 100%;
          max-width: 280px;
        }
        
        .features-grid {
          grid-template-columns: 1fr;
          gap: 1.5rem;
        }
        
        .gallery-grid {
          grid-template-columns: 1fr;
        }
        
        .section {
          padding: calc(var(--layout-spacing) * 1.5) 0;
        }
        
        .container {
          padding: 0 var(--layout-spacing);
        }
        
        .grid-cols-2,
        .grid-cols-3,
        .grid-cols-4 {
          grid-template-columns: 1fr;
        }
      }
      
      @media (max-width: 480px) {
        .hero-section {
          padding: 3rem 0;
        }
        
        .hero-title {
          font-size: calc(var(--font-size-h1) * 0.7);
        }
        
        .feature-item {
          padding: 1.5rem;
        }
        
        .contact-form {
          margin-top: 1.5rem;
        }
        
        .container {
          padding: 0 calc(var(--layout-spacing) * 0.75);
        }
      }
    `;
  }

  /**
   * Generate complete CSS from customization data
   */
  public generateStyles(): TemplateStyles {
    const cssVariables = this.generateCSSVariables();
    const baseStyles = this.generateBaseStyles();
    const sectionStyles = this.generateSectionStyles();
    const responsiveStyles = this.generateResponsiveStyles();

    return {
      cssVariables,
      baseStyles,
      sectionStyles,
      responsiveStyles
    };
  }

  /**
   * Generate HTML for a section based on its type
   */
  public generateSectionHTML(section: SectionData): string {
    switch (section.type) {
      case 'hero':
        return this.generateHeroSection(section);
      case 'features':
        return this.generateFeaturesSection(section);
      case 'about':
        return this.generateAboutSection(section);
      case 'services':
        return this.generateServicesSection(section);
      case 'gallery':
        return this.generateGallerySection(section);
      case 'contact':
        return this.generateContactSection(section);
      default:
        return this.generateGenericSection(section);
    }
  }

  private generateHeroSection(section: SectionData): string {
    const title = this.sanitizeTitle(section.content?.title || this.template.name || 'Welcome to Our Website');
    const subtitle = this.sanitizeContent(section.content?.text || this.template.description || 'Create something amazing with our platform');
    
    return `
      <section class="section hero-section" data-section-id="${this.sanitizeTitle(section.id)}">
        <div class="container">
          <h1 class="hero-title">${title}</h1>
          <p class="hero-subtitle">${subtitle}</p>
          <div class="hero-actions">
            <a href="#" class="btn btn-accent">Get Started</a>
            <a href="#" class="btn btn-secondary">Learn More</a>
          </div>
        </div>
      </section>
    `;
  }

  private generateFeaturesSection(section: SectionData): string {
    const title = this.sanitizeTitle(section.content?.title || 'Why Choose Us');
    const subtitle = this.sanitizeContent(section.content?.subtitle || 'Discover what makes us different');
    
    const features = section.content?.features || [
      { title: 'Fast & Reliable', description: 'Built with performance in mind', icon: '⚡' },
      { title: 'Customizable', description: 'Tailor every aspect to your needs', icon: '🎨' },
      { title: 'Professional', description: 'Modern design that looks great', icon: '✨' }
    ];

    const featuresHTML = features.map((feature: any) => `
      <div class="feature-item">
        <div class="feature-icon">${this.sanitizeTitle(feature.icon || '●')}</div>
        <h3>${this.sanitizeTitle(feature.title)}</h3>
        <p>${this.sanitizeContent(feature.description)}</p>
      </div>
    `).join('');

    return `
      <section class="section features-section" data-section-id="${this.sanitizeTitle(section.id)}">
        <div class="container">
          <h2 class="section-title">${title}</h2>
          <p class="section-subtitle">${subtitle}</p>
          <div class="features-grid">
            ${featuresHTML}
          </div>
        </div>
      </section>
    `;
  }

  private generateAboutSection(section: SectionData): string {
    const title = this.sanitizeTitle(section.content?.title || 'About Us');
    const text = this.sanitizeContent(section.content?.text || 'We are dedicated to providing exceptional service and innovative solutions that help your business grow and succeed in today\'s competitive market.');
    
    return `
      <section class="section about-section" data-section-id="${this.sanitizeTitle(section.id)}">
        <div class="container">
          <h2 class="section-title">${title}</h2>
          <p class="section-subtitle">${text}</p>
        </div>
      </section>
    `;
  }

  private generateServicesSection(section: SectionData): string {
    const title = this.sanitizeTitle(section.content?.title || 'Our Services');
    const text = this.sanitizeContent(section.content?.text || 'We offer a comprehensive range of services to meet your needs.');
    
    return `
      <section class="section services-section" data-section-id="${this.sanitizeTitle(section.id)}">
        <div class="container">
          <h2 class="section-title">${title}</h2>
          <p class="section-subtitle">${text}</p>
        </div>
      </section>
    `;
  }

  private generateGallerySection(section: SectionData): string {
    const title = this.sanitizeTitle(section.content?.title || 'Gallery');
    const subtitle = this.sanitizeContent(section.content?.subtitle || 'Take a look at our work');
    
    return `
      <section class="section gallery-section" data-section-id="${this.sanitizeTitle(section.id)}">
        <div class="container">
          <h2 class="section-title">${title}</h2>
          <p class="section-subtitle">${subtitle}</p>
          <div class="gallery-grid">
            <div class="gallery-item">Image 1</div>
            <div class="gallery-item">Image 2</div>
            <div class="gallery-item">Image 3</div>
            <div class="gallery-item">Image 4</div>
          </div>
        </div>
      </section>
    `;
  }

  private generateContactSection(section: SectionData): string {
    const title = this.sanitizeTitle(section.content?.title || 'Contact Us');
    const subtitle = this.sanitizeContent(section.content?.subtitle || 'Get in touch with us today');
    
    return `
      <section class="section contact-section" data-section-id="${this.sanitizeTitle(section.id)}">
        <div class="container">
          <h2 class="section-title">${title}</h2>
          <p class="section-subtitle">${subtitle}</p>
          <form class="contact-form">
            <div class="form-group">
              <label class="form-label" for="name">Name</label>
              <input type="text" id="name" class="form-input" placeholder="Your Name" required>
            </div>
            <div class="form-group">
              <label class="form-label" for="email">Email</label>
              <input type="email" id="email" class="form-input" placeholder="your@email.com" required>
            </div>
            <div class="form-group">
              <label class="form-label" for="message">Message</label>
              <textarea id="message" class="form-textarea" placeholder="Your message..." required></textarea>
            </div>
            <button type="submit" class="btn btn-primary">Send Message</button>
          </form>
        </div>
      </section>
    `;
  }

  private generateGenericSection(section: SectionData): string {
    const title = this.sanitizeTitle(section.content?.title || `${section.type} Section`);
    const text = this.sanitizeContent(section.content?.text || `This is a ${section.type} section with customized styling.`);
    
    return `
      <section class="section" data-section-id="${this.sanitizeTitle(section.id)}">
        <div class="container">
          <h2 class="section-title">${title}</h2>
          <p class="section-subtitle">${text}</p>
        </div>
      </section>
    `;
  }

  /**
   * Generate complete HTML document
   */
  public generateHTML(options: RenderOptions = {}): string {
    const { deviceMode = 'desktop', includeInteractivity = false, optimizeForExport = false } = options;
    const styles = this.generateStyles();
    
    const sections = this.customization.content.sections.length > 0 
      ? this.customization.content.sections
          .sort((a, b) => a.order - b.order)
          .map(section => this.generateSectionHTML(section))
          .join('')
      : this.generateDefaultSections();

    const cssVariablesString = Object.entries(styles.cssVariables)
      .map(([key, value]) => `${key}: ${value};`)
      .join('\n      ');

    const interactivityScripts = includeInteractivity ? this.generateInteractivityScripts() : '';

    return `<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>${this.sanitizeTitle(this.template.name || 'Website')}</title>
    <style>
      :root {
        ${cssVariablesString}
      }
      
      ${styles.baseStyles}
      ${styles.sectionStyles}
      ${styles.responsiveStyles}
    </style>
    ${optimizeForExport ? '<meta name="robots" content="noindex, nofollow">' : ''}
</head>
<body>
    ${sections}
    
    <footer class="footer-section">
      <div class="container">
        <p>&copy; 2025 ${this.sanitizeTitle(this.template.name || 'Your Website')}. All rights reserved.</p>
      </div>
    </footer>
    
    ${interactivityScripts}
</body>
</html>`;
  }

  private generateDefaultSections(): string {
    const defaultSections: SectionData[] = [
      { id: 'hero', type: 'hero', content: {}, order: 0 },
      { id: 'features', type: 'features', content: {}, order: 1 },
      { id: 'about', type: 'about', content: {}, order: 2 }
    ];

    return defaultSections.map(section => this.generateSectionHTML(section)).join('');
  }

  private generateInteractivityScripts(): string {
    return `
    <script>
      // Basic form handling
      const forms = document.querySelectorAll('form');
      forms.forEach(form => {
        form.addEventListener('submit', function(e) {
          e.preventDefault();
          alert('Form submitted! (This is a preview)');
        });
      });
      
      // Smooth scrolling for anchor links
      const links = document.querySelectorAll('a[href^="#"]');
      links.forEach(link => {
        link.addEventListener('click', function(e) {
          e.preventDefault();
          const target = document.querySelector(this.getAttribute('href'));
          if (target) {
            target.scrollIntoView({ behavior: 'smooth' });
          }
        });
      });
    </script>`;
  }

  /**
   * Content sanitization for XSS prevention
   */
  private sanitizeContent(content: string): string {
    // Configure DOMPurify for safe HTML generation
    return DOMPurify.sanitize(content, {
      ALLOWED_TAGS: ['b', 'i', 'em', 'strong', 'br', 'p'],
      ALLOWED_ATTR: [],
      KEEP_CONTENT: true,
      RETURN_DOM_FRAGMENT: false,
      RETURN_DOM: false
    });
  }

  private sanitizeTitle(title: string): string {
    // Sanitize and escape HTML entities for titles
    const sanitized = DOMPurify.sanitize(title, {
      ALLOWED_TAGS: [],
      ALLOWED_ATTR: [],
      KEEP_CONTENT: true
    });
    return sanitized.replace(/</g, '&lt;').replace(/>/g, '&gt;');
  }

  /**
   * Utility methods
   */
  private getSpacingValue(spacing: string): string {
    switch (spacing) {
      case 'compact': return '0.75rem';
      case 'relaxed': return '2rem';
      case 'loose': return '3rem';
      default: return '1.5rem';
    }
  }

  private hexToRgb(hex: string): string {
    const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
    if (!result) return '0, 0, 0';
    
    return [
      parseInt(result[1], 16),
      parseInt(result[2], 16),
      parseInt(result[3], 16)
    ].join(', ');
  }

  /**
   * Static method to create renderer instance
   */
  static create(template: WebsiteTemplate, customization: TemplateCustomization): TemplateRenderer {
    return new TemplateRenderer(template, customization);
  }
}

export default TemplateRenderer;