// Logo AI service client - calls Recraft API for AI-generated SVG logos
import { apiRequest } from "@/lib/queryClient";

export interface GenerateLogoInput {
  prompt: string;
  brandName?: string;
  colorHex?: string;
}

export interface GenerateLogoResult {
  svg: string;
}

/**
 * Generate an AI-powered SVG logo using Recraft API
 * @param input - Prompt and optional branding hints
 * @returns Promise with SVG string
 */
export async function generateLogoSvg(input: GenerateLogoInput): Promise<GenerateLogoResult> {
  const { prompt, brandName, colorHex } = input;
  
  if (!prompt || !prompt.trim()) {
    throw new Error('Prompt is required');
  }
  
  // Enhance prompt for better results
  const enhancedPrompt = enhancePrompt(prompt, brandName);
  
  try {
    // Step 1: Create prediction
    const createResponse = await apiRequest<{ predictionId: string; version: string }>(
      "POST",
      "/api/recraft/create",
      { prompt: enhancedPrompt }
    );
    
    const { predictionId } = createResponse;
    console.log(`[logo-ai] Created prediction: ${predictionId}`);
    
    // Step 2: Poll for result
    const svgUrl = await pollForResult(predictionId);
    
    // Step 3: Fetch SVG content
    const response = await fetch(svgUrl);
    if (!response.ok) {
      throw new Error(`Failed to fetch SVG: ${response.statusText}`);
    }
    
    const svg = await response.text();
    console.log(`[logo-ai] Successfully fetched SVG (${svg.length} bytes)`);
    
    return { svg };
    
  } catch (error) {
    console.error('[logo-ai] Generation failed:', error);
    throw error instanceof Error ? error : new Error('Failed to generate logo');
  }
}

/**
 * Enhance user prompt with best practices for logo generation
 */
function enhancePrompt(prompt: string, brandName?: string): string {
  const base = prompt.trim();
  const brandHint = brandName ? ` for ${brandName}` : '';
  
  // Add guidance for better icon-only logos
  return `${base}${brandHint}, minimal icon, geometric shapes, vector graphic, no text, transparent background, isolated logo icon`;
}

/**
 * Poll Recraft API for prediction result
 */
async function pollForResult(predictionId: string): Promise<string> {
  const timeoutAt = Date.now() + 210_000; // 3.5 minutes
  let pollCount = 0;
  
  while (Date.now() < timeoutAt) {
    pollCount++;
    
    const statusResponse = await apiRequest<{
      status: string;
      svgUrl?: string;
      error?: string;
      rawOutputShape?: string;
      debugOutput?: any;
    }>("GET", `/api/recraft/status/${predictionId}`);
    
    console.log(`[logo-ai] Poll #${pollCount}: ${statusResponse.status}`);
    
    if (statusResponse.status === 'succeeded') {
      if (statusResponse.svgUrl) {
        return statusResponse.svgUrl;
      } else {
        const debugMsg = statusResponse.debugOutput
          ? `No valid SVG URL found. Output shape: ${statusResponse.rawOutputShape}`
          : 'No valid SVG URL found in response';
        throw new Error(debugMsg);
      }
    }
    
    if (statusResponse.status === 'failed') {
      throw new Error(statusResponse.error || 'Generation failed on server');
    }
    
    // Still processing - wait before next poll
    const delay = statusResponse.status === 'queued' ? 3000 : 2000;
    await new Promise(resolve => setTimeout(resolve, delay));
  }
  
  throw new Error(`Generation timed out after ${pollCount} polls (3.5 minutes)`);
}
