import { addDoc, collection } from "firebase/firestore";
import { db } from "@/lib/firebase";
import { BrandKit } from "@/types/brand";

export async function saveBrandKit(userId: string, kit: BrandKit): Promise<string> {
  try {
    // Try Firestore first, but with timeout
    const timeoutPromise = new Promise((_, reject) => 
      setTimeout(() => reject(new Error('Save timeout')), 5000)
    );
    
    const savePromise = addDoc(collection(db, "brandKits"), {
      ...kit,
      userId,
      createdAt: new Date().toISOString(),
    });
    
    const docRef = await Promise.race([savePromise, timeoutPromise]);
    
    // Track brand kit creation for dashboard
    trackBrandKitCreation();
    
    return (docRef as any).id;
  } catch (error) {
    console.warn('Firestore save failed, using local storage fallback:', error);
    
    // Fallback to localStorage
    const id = `kit_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    const kitWithId = { ...kit, id, userId, createdAt: new Date().toISOString() };
    
    const savedKits = JSON.parse(localStorage.getItem('brandKits') || '[]');
    savedKits.push(kitWithId);
    localStorage.setItem('brandKits', JSON.stringify(savedKits));
    
    // Track brand kit creation for dashboard
    trackBrandKitCreation();
    
    return id;
  }
}

function trackBrandKitCreation() {
  try {
    const currentCount = parseInt(localStorage.getItem('brandKitCount') || '0', 10);
    localStorage.setItem('brandKitCount', (currentCount + 1).toString());
  } catch (error) {
    console.warn('Failed to track brand kit creation:', error);
  }
}