import JSZip from "jszip";
import { saveAs } from "file-saver";
import { BrandKit } from "@/types/brand";

function injectBackground(svg: string, color: string) {
  // insert a full-bleed rect right after the opening <svg ...>
  return svg.replace(/<svg([^>]+)>/, (_m, attrs) => `<svg${attrs}><rect width="100%" height="100%" fill="${color}"/>`);
}

function recolor(svg: string, fillColor: string, strokeColor?: string) {
  let out = svg;

  // fills (skip fill="none")
  out = out.replace(/fill="(?!none)[^"]*"/gi, `fill="${fillColor}"`);
  // strokes
  if (strokeColor) {
    out = out.replace(/stroke="[^"]*"/gi, `stroke="${strokeColor}"`);
  } else {
    // if no strokeColor provided, default to fillColor
    out = out.replace(/stroke="[^"]*"/gi, `stroke="${fillColor}"`);
  }

  // if any elements rely on currentColor, set color
  out = out.replace(/<svg([^>]+)>/, (_m, attrs) => `<svg${attrs} style="color:${fillColor}">`);
  return out;
}

// -------- helper (browser) ----------
async function svgToPng(svg: string, width: number, height: number): Promise<Blob> {
  const svgBlob = new Blob([svg], { type: "image/svg+xml" });
  const url = URL.createObjectURL(svgBlob);
  const img = new Image();
  img.decoding = "async";
  img.src = url;
  await img.decode();

  const canvas = document.createElement("canvas");
  canvas.width = width; canvas.height = height;
  const ctx = canvas.getContext("2d")!;
  // optional: clear to transparent
  ctx.clearRect(0,0,width,height);
  ctx.drawImage(img, 0, 0, width, height);

  URL.revokeObjectURL(url);
  return await new Promise<Blob>((resolve) => canvas.toBlob(b => resolve(b!), "image/png"));
}

export async function downloadBrandKitZip(
  kit: BrandKit,
  opts?: { includePNGs?: boolean; pngSizes?: number[] }
) {
  const includePNGs = opts?.includePNGs ?? true;           // default: on
  const sizes       = opts?.pngSizes ?? [1024, 512, 256];

  const zip = new JSZip();
  const base = (kit.businessName || "brand").replace(/\s+/g, "-").toLowerCase();

  const svgs = zip.folder(`${base}/svg`)!;
  kit.logos.forEach(v => svgs.file(v.filename, v.svg));

  if (includePNGs) {
    const pngFolder = zip.folder(`${base}/png`)!;
    for (const v of kit.logos) {
      for (const s of sizes) {
        const blob = await svgToPng(v.svg, s, s);
        pngFolder.file(v.filename.replace(/\.svg$/i, `-${s}.png`), blob);
      }
    }
  }

  // 2) monochrome (black) and reverse (white on primary bg)
  const mono = zip.folder(`${base}-monochrome`)!;
  const rev  = zip.folder(`${base}-reverse`)!;

  kit.logos.forEach(v => {
    const name = v.filename.replace(/\.svg$/i, "");
    // monochrome = black vector
    const monoSvg = recolor(v.svg, "#000000", "#000000");
    mono.file(`${name}-mono.svg`, monoSvg);

    // reverse = white on primary background
    const whiteSvg = recolor(v.svg, "#ffffff", "#ffffff");
    const reverseSvg = injectBackground(whiteSvg, kit.palette.primary);
    rev.file(`${name}-reverse.svg`, reverseSvg);
  });

  // palette preview
  const swatch = `
  <svg xmlns="http://www.w3.org/2000/svg" width="1200" height="220">
    <rect x="0" y="0" width="200" height="220" fill="${kit.palette.primary}"/>
    <rect x="200" y="0" width="200" height="220" fill="${kit.palette.secondary}"/>
    <rect x="400" y="0" width="200" height="220" fill="${kit.palette.accent}"/>
    <rect x="600" y="0" width="200" height="220" fill="${kit.palette.neutral}"/>
    <rect x="800" y="0" width="200" height="220" fill="${kit.palette.surface}"/>
    <rect x="1000" y="0" width="200" height="220" fill="${kit.palette.textPrimary}"/>
  </svg>`;
  svgs.file("palette.svg", swatch);

  // brand.json
  svgs.file("brand.json", JSON.stringify(kit, null, 2));

  const blob = await zip.generateAsync({ type: "blob" });
  saveAs(blob, `${base}-brand-kit.zip`);
  
  // Track download for dashboard stats
  trackDownload();
}

function trackDownload() {
  try {
    const currentCount = parseInt(localStorage.getItem('downloadCount') || '0', 10);
    localStorage.setItem('downloadCount', (currentCount + 1).toString());
  } catch (error) {
    console.warn('Failed to track download:', error);
  }
}