/**
 * Google Identity Services helper for authentication
 * Handles popup flow for Google Drive API access
 */

// TypeScript definitions for Google Identity Services
declare global {
  interface Window {
    google?: {
      accounts: {
        id: {
          initialize: (config: any) => void;
          prompt: () => void;
        };
        oauth2: {
          initTokenClient: (config: any) => any;
        };
      };
    };
  }
}

interface GoogleTokenResponse {
  access_token: string;
  expires_in: number;
  scope: string;
  token_type: string;
}

/**
 * Get Google access token for Drive API access
 * Uses popup flow with drive.file scope
 * @returns Promise<string> Access token for Google Drive API
 * @throws Error if authentication fails or user cancels
 */
export async function getGoogleAccessToken(): Promise<string> {
  // Check if Google Identity Services is loaded
  if (!window.google?.accounts?.oauth2) {
    throw new Error("Google Identity Services not loaded. Please try again.");
  }

  // Get Google Client ID from environment
  const clientId = import.meta.env.VITE_GOOGLE_CLIENT_ID;
  if (!clientId) {
    throw new Error("Google Client ID not configured. Please contact support.");
  }

  // Return Promise that resolves with access token
  return new Promise((resolve, reject) => {
    try {
      // Initialize OAuth2 token client
      const tokenClient = window.google!.accounts.oauth2.initTokenClient({
        client_id: clientId,
        scope: "https://www.googleapis.com/auth/drive.file",
        callback: (response: GoogleTokenResponse | { error: string; error_description?: string }) => {
          if ('error' in response) {
            // Handle OAuth errors
            if (response.error === 'popup_closed_by_user') {
              reject(new Error("Google sign-in was cancelled. Please try again."));
            } else {
              reject(new Error(response.error_description || response.error || "Google authentication failed"));
            }
            return;
          }

          // Successfully received access token
          if (response.access_token) {
            resolve(response.access_token);
          } else {
            reject(new Error("Failed to receive access token from Google"));
          }
        },
        error_callback: (error: any) => {
          console.error("Google OAuth error:", error);
          reject(new Error("Google authentication error. Please try again."));
        }
      });

      // Request access token (triggers popup)
      tokenClient.requestAccessToken();
      
    } catch (error) {
      console.error("Google auth initialization error:", error);
      reject(new Error("Failed to initialize Google authentication"));
    }
  });
}

/**
 * Check if Google Identity Services is ready
 * @returns boolean True if Google APIs are loaded and ready
 */
export function isGoogleAuthReady(): boolean {
  return !!(window.google?.accounts?.oauth2);
}

/**
 * Wait for Google Identity Services to load
 * @param timeout Maximum time to wait in milliseconds (default: 10000)
 * @returns Promise<boolean> Resolves true when ready, false on timeout
 */
export async function waitForGoogleAuth(timeout: number = 10000): Promise<boolean> {
  const startTime = Date.now();
  
  return new Promise((resolve) => {
    const checkReady = () => {
      if (isGoogleAuthReady()) {
        resolve(true);
        return;
      }
      
      if (Date.now() - startTime > timeout) {
        resolve(false);
        return;
      }
      
      setTimeout(checkReady, 100);
    };
    
    checkReady();
  });
}