import { FontOption } from '@/components/FontPicker';

export interface GenerateFontsResponse {
  fonts: string[];
  fallback?: boolean;
  error?: string;
}

/**
 * Generate 120 professional font options using Replicate API
 */
export async function generateFonts(): Promise<string[]> {
  const response = await fetch('/api/generate-fonts', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  if (!response.ok) {
    throw new Error(`Failed to generate fonts: ${response.statusText}`);
  }

  const data: GenerateFontsResponse = await response.json();
  return data.fonts;
}

/**
 * Categorize a font name into a group based on common font characteristics
 */
function categorizeFontByName(fontName: string): FontOption["group"] {
  const name = fontName.toLowerCase();
  
  // Script/Handwriting fonts
  if (name.includes('script') || name.includes('handwriting') || name.includes('cursive') || 
      name.includes('dancing') || name.includes('great vibes') || name.includes('pacifico') || 
      name.includes('sacramento') || name.includes('kaushan') || name.includes('shadows') ||
      name.includes('amatic') || name.includes('indie flower') || name.includes('permanent marker') ||
      name.includes('courgette') || name.includes('calligraphy')) {
    return "Script";
  }
  
  // Display fonts
  if (name.includes('display') || name.includes('bebas') || name.includes('oswald') || 
      name.includes('anton') || name.includes('fjalla') || name.includes('righteous') ||
      name.includes('orbitron') || name.includes('exo') || name.includes('bungee') ||
      name.includes('varela') || name.includes('play') || name.includes('condensed') ||
      name.includes('black') && name.includes('ops') || name.includes('impact')) {
    return "Display";
  }
  
  // Slab serif fonts
  if (name.includes('slab') || name.includes('arvo') || name.includes('roboto slab') ||
      name.includes('noto serif display') || name.includes('alegreya') || name.includes('tinos') ||
      name.includes('rockwell') || name.includes('clarendon') || name.includes('egyptian')) {
    return "Slab";
  }
  
  // Serif fonts
  if (name.includes('serif') || name.includes('playfair') || name.includes('merriweather') ||
      name.includes('cormorant') || name.includes('crimson') || name.includes('lora') ||
      name.includes('spectral') || name.includes('vollkorn') || name.includes('domine') ||
      name.includes('pt serif') || name.includes('times') || name.includes('georgia') ||
      name.includes('garamond') || name.includes('baskerville') || name.includes('caslon') ||
      name.includes('minion') || name.includes('charter')) {
    return "Serif";
  }
  
  // Mono fonts
  if (name.includes('mono') || name.includes('code') || name.includes('consolas') ||
      name.includes('courier') || name.includes('fira') && name.includes('code') ||
      name.includes('source code') || name.includes('jetbrains') || name.includes('ubuntu mono') ||
      name.includes('inconsolata') || name.includes('liberation mono')) {
    return "Mono";
  }
  
  // Default to Sans if no other category matches
  return "Sans";
}

/**
 * Convert font name to CSS font-family format
 */
function createFontFamily(fontName: string, group: FontOption["group"]): string {
  const quotedName = `"${fontName}"`;
  
  switch (group) {
    case "Sans":
      return `${quotedName}, sans-serif`;
    case "Serif":
    case "Slab":
      return `${quotedName}, serif`;
    case "Script":
    case "Display":
      return `${quotedName}, cursive`;
    case "Mono":
      return `${quotedName}, monospace`;
    default:
      return `${quotedName}, sans-serif`;
  }
}

/**
 * Convert array of font names to FontOption array with proper categorization
 */
export function convertToFontOptions(fontNames: string[]): FontOption[] {
  return fontNames.map((fontName) => {
    const group = categorizeFontByName(fontName);
    const family = createFontFamily(fontName, group);
    
    return {
      label: fontName,
      family: family,
      group: group,
    };
  });
}

/**
 * Get generated fonts formatted as FontOptions
 */
export async function getGeneratedFontOptions(): Promise<FontOption[]> {
  try {
    const fontNames = await generateFonts();
    return convertToFontOptions(fontNames);
  } catch (error) {
    console.error('Failed to generate fonts, using empty array:', error);
    return [];
  }
}