import { db, storage } from "@/utils/firebase-templates";
import { doc, getDoc, collection, getDocs, query, where } from "firebase/firestore";
import { getDownloadURL, ref } from "firebase/storage";

// ---- Types your page wants ----
export type LogoComposer = {
  id: string;
  name: string;
  previewUrl?: string;
  svg: string; // inline SVG with {Brand_Name} {Tagline} {Est_Year} tokens
  defaultFields: Record<string, string>;
  fieldOrder?: string[];
  defaultColors: { primary: string; secondary: string; accent: string };
};

// Legacy type for gallery compatibility
export type FirebaseLogoComposer = {
  id: string;
  name: string;
  description: string;
  tags: string[];
  version: number;
  status: "active" | "draft" | "archived";
  assets: {
    svgPath: string;
    previewPngPath: string;
    svgDownloadUrl: string;
    previewDownloadUrl: string;
  };
  defaults: {
    brandName: string;
    tagline: string;
    estYear: string;
  };
  createdBy: string;
  createdAt: any;
  updatedAt: any;
};

// ---- helpers ----
async function resolveToHttpUrl(maybeUrl?: string) {
  if (!maybeUrl) return undefined;

  // If it's an http(s) URL already, use it
  if (/^https?:\/\//i.test(maybeUrl)) return maybeUrl;

  // If it's a gs:// path, convert to http download URL
  if (maybeUrl.startsWith("gs://")) {
    return await getDownloadURL(ref(storage, maybeUrl));
  }

  // Otherwise treat as a bucket object path (e.g. templates/logos/.../source.svg)
  return await getDownloadURL(ref(storage, maybeUrl));
}

export async function fetchText(u: string): Promise<string> {
  if (!u) return "";
  try {
    // Use proxy for Firebase URLs to avoid CORS issues
    if (/^https:\/\/(firebasestorage|storage|firestore)\.googleapis\.com\//i.test(u)) {
      const proxyUrl = `/api/firebase-proxy?url=${encodeURIComponent(u)}`;
      const res = await fetch(proxyUrl, { cache: "no-store" });
      if (!res?.ok) return "";
      return await res.text();
    } else {
      // Direct fetch for non-Firebase URLs
      const res = await fetch(u, { cache: "no-store" });
      if (!res?.ok) return "";
      return await res.text();
    }
  } catch { return ""; }
}

export async function readDoc(collectionName: string, id: string) {
  const snap = await getDoc(doc(db, collectionName, id));
  if (!snap.exists()) return null;
  const d = snap.data() as any;

  // prefer downloadURLs.*; fall back to storagePaths.*; normalize both to http URLs
  const svgUrl = await resolveToHttpUrl(
    d?.downloadURLs?.svg ?? d?.storagePaths?.svg
  );

  const previewUrl = await resolveToHttpUrl(
    d?.downloadURLs?.preview ??
    d?.downloadURLs?.raster ??
    d?.storagePaths?.preview ??
    d?.storagePaths?.raster
  ) || svgUrl;

  // Inline SVG string (optional)
  let svgString = await fetchText(svgUrl || "");
  
  // Add token placeholders if missing (for legacy SVGs)
  if (svgString && !svgString.includes("{Brand_Name}")) {
    console.log(`🔧 Adding missing tokens to SVG for template ${id}`);
    svgString = svgString.replace(
      "</svg>",
      `<g id="ibrandbiz-tokens" style="opacity:0">
         <text>{Brand_Name}</text><text>{Tagline}</text><text>{Est_Year}</text>
       </g></svg>`
    );
  }

  // Defaults mapping (tolerant to different field names)
  const defaultFields = {
    Brand_Name: d?.defaults?.Brand_Name ?? d?.defaults?.brandName ?? d?.brandName ?? "Your Brand",
    Tagline:    d?.defaults?.Tagline    ?? d?.defaults?.tagline    ?? d?.tagline    ?? "Your Tagline",
    Est_Year:   d?.defaults?.Est_Year   ?? d?.defaults?.estYear    ?? d?.estYear    ?? "2025",
  };

  // Colors (fallbacks)
  const defaultColors = {
    primary:   d?.defaultColors?.primary   ?? "#222222",
    secondary: d?.defaultColors?.secondary ?? "#666666",
    accent:    d?.defaultColors?.accent    ?? "#000000",
  };

  const fieldOrder = d?.fieldOrder ?? ["Brand_Name", "Tagline", "Est_Year"];
  const name = d?.name || id;

  return {
    id,
    name,
    previewUrl,            // used for the image preview
    svg: svgString || "",  // keep empty if not fetchable; page won't crash
    defaultFields,
    fieldOrder,
    defaultColors,
  } satisfies LogoComposer;
}

/**
 * getTemplateById - Returns Firebase template in the structure LogoCustomizerPage expects
 * Tries 'templates' first, then 'logo_templates'.
 * Also tries a short id if incoming id is prefixed (e.g., "logo-wordmark-").
 */
export async function getTemplateById(id: string) {
  try {
    // Query both collections: templates (new) and logo_templates (legacy)
    const collections = ["templates", "logo_templates"];
    const searchIds = [id];
    if (id.startsWith("logo-wordmark-")) {
      searchIds.push(id.replace("logo-wordmark-", ""));
    }
    
    for (const collectionName of collections) {
      try {
        const collectionRef = collection(db, collectionName);
        const querySnapshot = await getDocs(collectionRef);
        
        // Process documents with async download URL generation
        for (const doc of querySnapshot.docs) {
          const data = doc.data();
          const composerId = data.templateId || doc.id;
          
          // Check if this is the composer we're looking for
          if (!searchIds.includes(composerId)) continue;
          
          // normalize all possible sources to http URLs
          const svgDownloadUrl = await resolveToHttpUrl(
            data.downloadURLs?.svg ?? data.storagePaths?.svg
          ) || "";
          
          const previewDownloadUrl = await resolveToHttpUrl(
            data.downloadURLs?.preview ??
            data.downloadURLs?.raster ??
            data.storagePaths?.preview ??
            data.storagePaths?.raster
          ) || svgDownloadUrl;
          
          // Return in the structure LogoCustomizerPage expects (FirebaseLogoComposer structure)
          const result = {
            id: composerId,
            name: data.name || composerId,
            assets: {
              svgDownloadUrl,
              previewDownloadUrl
            },
            defaults: {
              brandName: data.defaults?.brandName || data.brandName || "Your Brand",
              tagline: data.defaults?.tagline || data.tagline || "Your Tagline",
              estYear: data.defaults?.estYear || data.estYear || "2025"
            }
          };
          
          return result;
        }
      } catch (collectionError) {
        console.warn(`Failed to read ${collectionName} collection:`, collectionError);
      }
    }
    
    return null;
  } catch (error) {
    console.error("Failed to get template by ID:", error);
    return null;
  }
}

// Legacy gallery function for LogoComposers.tsx compatibility
export async function fetchLogoTemplates(options: {
  onlyPublic?: boolean;
  useStatusFilter?: boolean;
  status?: "active" | "draft" | "archived";
}): Promise<{ items: FirebaseLogoComposer[] }> {
  try {
    const allComposers: FirebaseLogoComposer[] = [];
    const seenIds = new Set<string>();
    
    // Query both collections: templates (new) and logo_templates (legacy)
    const collections = ["templates", "logo_templates"];
    
    for (const collectionName of collections) {
      try {
        const collectionRef = collection(db, collectionName);
        
        // Build query with status filter if enabled
        let composerQuery = collectionRef;
        if (options.useStatusFilter && options.status) {
          composerQuery = query(collectionRef, where("status", "==", options.status));
        }
        
        const querySnapshot = await getDocs(composerQuery);
        
        // Process documents with async download URL generation
        for (const doc of querySnapshot.docs) {
          const data = doc.data();
          const composerId = data.templateId || doc.id;
          
          // Skip duplicates across collections
          if (seenIds.has(composerId)) continue;
          seenIds.add(composerId);
          
          // normalize all possible sources to http URLs
          const svgDownloadUrl = await resolveToHttpUrl(
            data.downloadURLs?.svg ?? data.storagePaths?.svg
          ) || "";
          const previewDownloadUrl = await resolveToHttpUrl(
            data.downloadURLs?.preview ??
            data.downloadURLs?.raster ??
            data.storagePaths?.preview ??
            data.storagePaths?.raster
          ) || svgDownloadUrl;
          
          // Map SN's template structure to the expected format
          const composer: FirebaseLogoComposer = {
            id: composerId,
            name: data.name || composerId,
            description: data.description || "",
            tags: data.tags || [],
            version: data.version || 1,
            status: data.status || "active",
            assets: {
              svgPath: data.storagePaths?.svg || "",
              previewPngPath: data.storagePaths?.preview || data.storagePaths?.raster || "",
              svgDownloadUrl,
              previewDownloadUrl
            },
            defaults: {
              brandName: data.defaults?.brandName || data.brandName || "Your Brand",
              tagline: data.defaults?.tagline || data.tagline || "Your Tagline",
              estYear: data.defaults?.estYear || data.estYear || "2025"
            },
            createdBy: data.ownerId || data.createdBy || "",
            createdAt: data.createdAt,
            updatedAt: data.updatedAt
          };
          
          allComposers.push(composer);
        }
      } catch (collectionError) {
        console.warn(`Failed to read ${collectionName} collection:`, collectionError);
      }
    }
    
    return { items: allComposers };
  } catch (error) {
    console.error("Failed to load Firebase composers:", error);
    return { items: [] };
  }
}