export type RenderOpts = {
  width: number;
  bg?: string | null;
  withIcon: boolean;
  withText: boolean;
  layout: "side-by-side" | "top-bottom";
  iconSize?: number;
  textSize?: number;
  textColor: string;
  fontFamily: string;
  name: string;
  svgUrl?: string | null;
  pad?: number;
};

export async function renderLogoPNG(opts: RenderOpts): Promise<Blob> {
  const { width, bg=null, withIcon, withText, layout, iconSize=300, textSize=160, textColor, fontFamily, name, svgUrl, pad=64 } = opts;
  const height = Math.round(width * (layout === "side-by-side" ? 0.5 : 0.75));
  const canvas = document.createElement("canvas");
  const ctx = canvas.getContext("2d")!;
  canvas.width = width; canvas.height = height;
  if (bg) { ctx.fillStyle = bg; ctx.fillRect(0,0,width,height); } else { ctx.clearRect(0,0,width,height); }
  const cx = width / 2, cy = height / 2;

  let iconDrawn = false;
  if (withIcon && svgUrl) {
    const img = new Image(); img.crossOrigin = "anonymous";
    await new Promise<void>(res => { img.onload=()=>res(); img.onerror=()=>res(); img.src = svgUrl; });
    if (layout === "side-by-side") {
      const total = iconSize + (withText ? pad : 0);
      const x = withText ? cx - total/2 : cx - iconSize/2;
      ctx.drawImage(img, x, cy - iconSize/2, iconSize, iconSize);
    } else {
      ctx.drawImage(img, cx - iconSize/2, cy - (withText ? (iconSize/2 + pad/2) : iconSize/2), iconSize, iconSize);
    }
    iconDrawn = true;
  }

  if (withText) {
    ctx.fillStyle = textColor;
    ctx.font = `bold ${textSize}px ${fontFamily}`;
    ctx.textBaseline = "alphabetic";
    ctx.textAlign = layout === "side-by-side" ? "left" : "center";
    const m = ctx.measureText(name);
    if (layout === "side-by-side") {
      const x = iconDrawn ? cx + (pad/2) : cx - m.width/2;
      const y = cy + (iconDrawn ? iconSize/2 : textSize/2);
      ctx.fillText(name, x, y);
    } else {
      ctx.fillText(name, cx, cy + iconSize/2 + textSize/2);
    }
  }
  return await new Promise<Blob>(r => canvas.toBlob(b => r(b!), "image/png"));
}

export async function renderLogoJPG(opts: Omit<RenderOpts,"bg"> & { bg: string; quality?: number }): Promise<Blob> {
  const blobPng = await renderLogoPNG({ ...opts, bg: opts.bg });
  const url = URL.createObjectURL(blobPng);
  const img = new Image();
  await new Promise<void>(res => { img.onload=()=>res(); img.onerror=()=>res(); img.src=url; });
  const canvas = document.createElement("canvas");
  canvas.width = img.width; canvas.height = img.height;
  const ctx = canvas.getContext("2d")!; ctx.drawImage(img,0,0);
  URL.revokeObjectURL(url);
  return await new Promise<Blob>(r => canvas.toBlob(b => r(b!), "image/jpeg", opts.quality ?? 0.92));
}