export type ComposeSvgOpts = {
  name: string;
  fontFamily: string;
  textSize: number;
  textColor: string;
  layout: "side-by-side" | "top-bottom";
  iconSize: number;
  pad?: number;
  iconSvgText: string;
  background?: string | null;
  textColorOverride?: string;
};

function parseViewBox(svg: string) {
  const m = svg.match(/viewBox="([\d.\-]+)\s+([\d.\-]+)\s+([\d.\-]+)\s+([\d.\-]+)"/i);
  if (!m) return { x: 0, y: 0, w: 1000, h: 1000 };
  return { x: +m[1], y: +m[2], w: +m[3], h: +m[4] };
}
function extractInner(svg: string) {
  const s = svg.indexOf(">") + 1, e = svg.lastIndexOf("</svg>");
  return s > 0 && e > s ? svg.slice(s, e) : svg;
}
function measure(text: string, fontFamily: string, px: number) {
  const c = document.createElement("canvas");
  const ctx = c.getContext("2d")!;
  ctx.font = `bold ${px}px ${fontFamily}, system-ui, sans-serif`;
  return ctx.measureText(text).width;
}
function esc(x: string) { return x.replace(/[&<>"']/g, c => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&apos;'} as any)[c]); }

export function composeLogoSvg(opts: ComposeSvgOpts) {
  const { name, fontFamily, textSize, textColor, layout, iconSize, pad=48, iconSvgText, background=null, textColorOverride } = opts;
  const vb = parseViewBox(iconSvgText);
  const inner = extractInner(iconSvgText);
  const sx = iconSize / vb.w, sy = iconSize / vb.h;
  const tw = measure(name, fontFamily, textSize);
  const th = textSize;

  let w: number, h: number;
  if (layout === "side-by-side") { w = iconSize + pad + tw; h = Math.max(iconSize, th * 1.2); }
  else { w = Math.max(iconSize, tw); h = iconSize + pad + th * 1.2; }

  const bg = background ? `<rect x="0" y="0" width="${w}" height="${h}" fill="${background}"/>` : "";
  const icon =
    `<g transform="translate(${layout==='side-by-side'?0:(w-iconSize)/2},${layout==='side-by-side'?(h-iconSize)/2:0}) scale(${sx},${sy}) translate(${-vb.x},${-vb.y})">${inner}</g>`;
  const fill = textColorOverride ?? textColor;
  const text = layout === "side-by-side"
    ? `<text x="${iconSize+pad}" y="${(h+th*0.85)/2}" font-family="${fontFamily}, system-ui, sans-serif" font-weight="700" font-size="${textSize}" fill="${fill}">${esc(name)}</text>`
    : `<text x="${w/2}" y="${iconSize+pad+th*0.85}" text-anchor="middle" font-family="${fontFamily}, system-ui, sans-serif" font-weight="700" font-size="${textSize}" fill="${fill}">${esc(name)}</text>`;

  return `<?xml version="1.0" encoding="UTF-8"?>
<svg xmlns="http://www.w3.org/2000/svg" width="${w}" height="${h}" viewBox="0 0 ${w} ${h}">
  ${bg}
  ${icon}
  ${text}
</svg>`;
}