import type { BrandKit } from "../../types/brand";

export interface ExportAssembleInput {
  title: string;                 // Plan title
  subtitle?: string;             // Optional subtitle
  coverDataUrl?: string;         // PNG data URL for cover (from CoverDesigner)
  sections: Array<{ id: string; title: string; content: string }>; // current plan sections in order
  appendixImages?: Array<{ title: string; dataUrl: string }>;      // any extra chart/table snapshots
  includeTOC?: boolean;          // defaults true
}

export interface AssembledDocument {
  markdown: string;              // fully assembled MD
  meta: { words: number; images: number };
}

function sanitize(s: string){
  return (s || "").replace(/\u0000/g, "");
}

export function assembleMarkdownDoc(input: ExportAssembleInput, brand?: BrandKit): AssembledDocument {
  const parts: string[] = [];
  const title = sanitize(input.title || "Business Plan");
  const subtitle = sanitize(input.subtitle || "");

  // Cover
  parts.push(`# ${title}`);
  if (subtitle) parts.push(`**${subtitle}**`);
  if (brand?.name) parts.push(`\n*Prepared for **${brand.name}***\n`);
  if (input.coverDataUrl) {
    parts.push(`\n![Cover](${input.coverDataUrl})\n`);
  }

  // TOC
  if (input.includeTOC !== false) {
    parts.push(`\n## Table of Contents`);
    input.sections.forEach((s, idx)=> {
      const anchor = s.title.toLowerCase().replace(/[^a-z0-9]+/g, "-").replace(/(^-|-$)/g, "");
      parts.push(`${idx+1}. [${sanitize(s.title)}](#${anchor})`);
    });
    if (input.appendixImages?.length) parts.push(`${input.sections.length+1}. [Appendix](#appendix)`);
  }

  // Sections
  input.sections.forEach((s)=>{
    const heading = sanitize(s.title || "Section");
    parts.push(`\n\n## ${heading}`);
    // content may already contain markdown + embedded data URLs from snapshots
    parts.push(sanitize(s.content || "_No content yet._"));
  });

  // Appendix (snapshots)
  if (input.appendixImages?.length){
    parts.push(`\n\n## Appendix`);
    input.appendixImages.forEach(img=>{
      const t = sanitize(img.title || "Figure");
      parts.push(`\n### ${t}`);
      parts.push(`![${t}](${img.dataUrl})`);
    });
  }

  const markdown = parts.join("\n\n");
  const words = markdown.split(/\s+/).filter(Boolean).length;
  const images = (markdown.match(/!\[/g) || []).length;
  return { markdown, meta: { words, images } };
}