// src/services/brand/palette.ts
export type PaletteKey =
  | "primary"
  | "secondary"
  | "accent"
  | "highlight"   // NEW
  | "neutral"
  | "surface"
  | "textLight"
  | "textDark";

export type Palette = Record<PaletteKey, string>;

export type PalettePreset =
  | "warm"
  | "cool"
  | "earth"
  | "sepia"
  | "pastel"
  | "neon";

export function rollHex(): string {
  return `#${Math.floor(Math.random() * 0xffffff)
    .toString(16)
    .padStart(6, "0")}`;
}

export function rollAll(prev?: Palette): Palette {
  return {
    primary: rollHex(),
    secondary: rollHex(),
    accent: rollHex(),
    highlight: rollHex(),                  // NEW
    neutral: prev?.neutral ?? "#f3f4f6",
    surface: prev?.surface ?? "#ffffff",
    textLight: prev?.textLight ?? "#f8fafc",
    textDark: prev?.textDark ?? "#111827",
  };
}

export function rollOne(p: Palette, key: PaletteKey): Palette {
  return { ...p, [key]: rollHex() };
}

/* ----------------- PRESETS ----------------- */
type Seed = {
  primary: string[];
  secondary: string[];
  accent: string[];
  highlight: string[];     // NEW
  neutral: string;
  surface: string;
  textLight: string;
  textDark: string;
};

const PRESET_SEEDS: Record<PalettePreset, Seed> = {
  warm: {
    primary: ["#d97706", "#ea580c", "#dc2626"],
    secondary: ["#b91c1c", "#c2410c", "#a16207"],
    accent: ["#f59e0b", "#fb7185", "#f97316"],
    highlight: ["#fde047", "#fbbf24", "#fb923c"],   // warm highlight
    neutral: "#f5f5f5",
    surface: "#ffffff",
    textLight: "#f8fafc",
    textDark: "#111827",
  },
  cool: {
    primary: ["#2563eb", "#0ea5e9", "#0891b2"],
    secondary: ["#1d4ed8", "#06b6d4", "#10b981"],
    accent: ["#60a5fa", "#22d3ee", "#34d399"],
    highlight: ["#a78bfa", "#93c5fd", "#67e8f9"],   // cool highlight
    neutral: "#eef2f7",
    surface: "#ffffff",
    textLight: "#f8fafc",
    textDark: "#0f172a",
  },
  earth: {
    primary: ["#7c5e3a", "#8c6d4f", "#6d4c32"],
    secondary: ["#a16207", "#3f6212", "#5b3a29"],
    accent: ["#86efac", "#22c55e", "#65a30d"],
    highlight: ["#eab308", "#a3e635", "#d97706"],
    neutral: "#f3f4f1",
    surface: "#ffffff",
    textLight: "#fafaf9",
    textDark: "#1f2937",
  },
  sepia: {
    primary: ["#b08968", "#a98467", "#7f5539"],
    secondary: ["#ddb892", "#e6ccb2", "#9c6644"],
    accent: ["#e3caa5", "#ffedd5", "#b08968"],
    highlight: ["#fde68a", "#f5deb3", "#fcd34d"],
    neutral: "#f7efe7",
    surface: "#fffaf0",
    textLight: "#fefcf8",
    textDark: "#2b2b2b",
  },
  pastel: {
    primary: ["#93c5fd", "#c7d2fe", "#fbcfe8"],
    secondary: ["#a7f3d0", "#fde68a", "#fca5a5"],
    accent: ["#f5d0fe", "#bfdbfe", "#bbf7d0"],
    highlight: ["#fff7ae", "#ffd6e7", "#d8fff2"],
    neutral: "#f8fafc",
    surface: "#ffffff",
    textLight: "#f8fafc",
    textDark: "#111827",
  },
  neon: {
    primary: ["#22d3ee", "#a3e635", "#f59e0b"],
    secondary: ["#f43f5e", "#8b5cf6", "#14b8a6"],
    accent: ["#f472b6", "#34d399", "#60a5fa"],
    highlight: ["#facc15", "#22c55e", "#38bdf8"],
    neutral: "#0b1020",
    surface: "#0f172a",
    textLight: "#e5e7eb",
    textDark: "#f8fafc",
  },
};

function pick<T>(arr: T[]): T {
  return arr[Math.floor(Math.random() * arr.length)];
}

export function makePresetPalette(preset: PalettePreset): Palette {
  const s = PRESET_SEEDS[preset];
  return {
    primary: pick(s.primary),
    secondary: pick(s.secondary),
    accent: pick(s.accent),
    highlight: pick(s.highlight),       // NEW
    neutral: s.neutral,
    surface: s.surface,
    textLight: s.textLight,
    textDark: s.textDark,
  };
}

export function applyPresetToPalette(
  preset: PalettePreset,
  current: Palette,
  locks: Record<PaletteKey, boolean>
): Palette {
  const next = { ...current };
  const seed = makePresetPalette(preset);
  (Object.keys(seed) as PaletteKey[]).forEach((k) => {
    if (!locks[k]) next[k] = seed[k];
  });
  return next;
}

export function rollWithinPreset(
  preset: PalettePreset,
  current: Palette,
  locks: Record<PaletteKey, boolean>
): Palette {
  const s = PRESET_SEEDS[preset];
  const next = { ...current };
  const maybeSet = (k: PaletteKey, pool?: string[]) => {
    if (locks[k]) return;
    if (!pool?.length) return;
    next[k] = pick(pool);
  };
  maybeSet("primary", s.primary);
  maybeSet("secondary", s.secondary);
  maybeSet("accent", s.accent);
  maybeSet("highlight", s.highlight);      // NEW

  if (!locks.neutral) next.neutral = s.neutral;
  if (!locks.surface) next.surface = s.surface;
  if (!locks.textLight) next.textLight = s.textLight;
  if (!locks.textDark) next.textDark = s.textDark;

  return next;
}