/**
 * Post-authentication redirect logic for IBrandBiz
 * Handles redirecting users based on their subscription status after successful authentication
 */
import { toast } from "sonner";

export interface AuthRedirectOptions {
  isPaid: boolean;
  newUser?: boolean;
  email?: string;
  navigate: (path: string) => void;
}

/**
 * Handles post-authentication redirect logic
 * @param options - Object containing isPaid status and navigate function
 */
export function handleAuthSuccess({ isPaid, newUser, email, navigate }: AuthRedirectOptions): void {
  // All users go to dashboard after successful login
  if (isPaid) {
    toast.success(`Welcome back${email ? `, ${email}` : ""}! Your Pro subscription is active 🚀`);
  } else {
    toast.success(`Welcome${newUser ? '' : ' back'}${email ? `, ${email}` : ""}! Let's build your brand 🚀`);
  }
  // Redirect all users to dashboard
  navigate('/dashboard');
}

/**
 * Determines if user is on paid plan or has owner access
 * Owner access: Checks against configured owner emails
 * In Phase 2, this will also check actual subscription status from backend/Firebase
 */
export function getUserPaidStatus(userEmail?: string): boolean {
  // Check if user has owner access via environment variable
  if (userEmail && isOwnerEmail(userEmail)) {
    return true;
  }
  
  // Phase 2: Will implement actual subscription checking here
  // For now, non-owner users don't have paid access
  return false;
}

/**
 * Checks if an email is in the list of configured owner emails
 */
export function isOwnerEmail(email: string): boolean {
  if (!email) return false;
  
  // Get owner emails from environment variable
  const ownerEmails = import.meta.env.VITE_IBRANDBIZ_OWNER_EMAILS || '';
  if (!ownerEmails) {
    // Development fallback - replace with actual environment variable in production
    if (import.meta.env.MODE === 'development') {
      console.log('⚠️ VITE_IBRANDBIZ_OWNER_EMAILS not configured. Add owner emails to environment variables.');
    }
    return false;
  }
  
  // Parse comma-separated list and normalize emails
  const ownerList = ownerEmails
    .split(',')
    .map(e => e.trim().toLowerCase())
    .filter(e => e.length > 0);
  
  return ownerList.includes(email.toLowerCase());
}