// src/services/ai/slogan.ts
export type SloganRequest = {
  brandName: string;
  description?: string;
  tone?: "Professional" | "Playful" | "Bold" | "Minimal" | "Luxury" | "Friendly";
  industry?: string;
  audience?: string;
  maxResults?: number; // default 8
};

export type SloganResponse = {
  suggestions: string[];
};

const uniq = (arr: string[]) => {
  const seen = new Set<string>();
  return arr
    .map(s => s.trim().replace(/\s+/g, " "))
    .filter(s => {
      const k = s.toLowerCase();
      if (seen.has(k)) return false;
      seen.add(k);
      return s.length > 0;
    });
};

const postprocess = (arr: string[]) => {
  // normalize punctuation, strip quotes, title-case brand name runs, etc.
  return uniq(
    arr.map(s =>
      s
        .replace(/^["""']+|["""']+$/g, "")
        .replace(/\s+([?.!,;:])/g, "$1")
        .replace(/\s{2,}/g, " ")
        .trim()
    )
  );
};

export async function generate(req: SloganRequest): Promise<SloganResponse> {
  const body = {
    brandName: req.brandName,
    description: req.description ?? "",
    tone: req.tone ?? "Professional",
    industry: req.industry ?? "",
    audience: req.audience ?? "",
    maxResults: req.maxResults ?? 8,
  };

  // Call your backend (Next.js API or Express). Do NOT call OpenAI from the browser.
  const res = await fetch("/api/slogan", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(body),
  });

  if (!res.ok) {
    throw new Error(`Slogan API ${res.status}`);
  }

  const data = (await res.json()) as { suggestions?: string[] };
  const cleaned = postprocess(data.suggestions ?? []);

  // Safety net: if model gave too few, synthesize variations locally so UI never looks broken.
  if (cleaned.length < (body.maxResults || 8)) {
    const base =
      cleaned[0] ||
      `${body.brandName}: ${body.tone === "Playful" ? "Make AI fun." : "Build with intelligence."}`;

    const riffs = [
      `${body.brandName}. Build your own AI.`,
      `${body.brandName} — From idea to AI.`,
      `Teach your AI. Launch your future.`,
      `Your AI. Your rules.`,
      `Turn curiosity into capability.`,
      `From zero to AI hero.`,
      `Make smart things. Smarter.`,
      `Own your intelligence.`,
      `Ship your own AI.`,
    ];

    cleaned.push(...riffs);
  }

  // Trim to requested count
  return { suggestions: uniq(cleaned).slice(0, body.maxResults || 8) };
}