/**
 * Google Docs export service for business plans
 * Handles uploading plans to Google Drive and converting to Google Docs format
 */

import { PlanResponse } from "./plan";
import { auth } from "@/lib/firebase";

export interface GoogleDocExportResponse {
  fileId: string;
  fileUrl: string;
  fileName: string;
}

/**
 * Export a business plan to Google Docs
 * @param plan - The generated business plan to export
 * @param businessName - Business name for filename and title
 * @param accessToken - Google access token with drive.file scope
 * @returns Promise<GoogleDocExportResponse> File details of created Google Doc
 * @throws Error with "PAYWALL" message for paywall errors
 */
export async function exportPlanToGoogleDoc(
  plan: PlanResponse,
  businessName: string,
  accessToken: string
): Promise<GoogleDocExportResponse> {
  // Get Firebase ID token for authentication
  const user = auth.currentUser;
  const token = user ? await user.getIdToken() : null;
  
  // Prepare headers with authentication
  const headers: Record<string, string> = {
    "Content-Type": "application/json"
  };
  
  // Add authentication token if available
  if (token) {
    headers["Authorization"] = `Bearer ${token}`;
  }
  
  // Add development bypass header in development mode
  const isDevelopment = import.meta.env.MODE === 'development';
  if (isDevelopment) {
    headers["x-dev-bypass"] = "development";
  }

  // Make request to backend Google Docs export endpoint
  const response = await fetch("/api/ai/plan/export/google-doc", {
    method: "POST",
    headers,
    body: JSON.stringify({ 
      plan, 
      businessName, 
      accessToken 
    }),
  });

  if (!response.ok) {
    // Handle paywall error
    if (response.status === 402) {
      throw new Error("PAYWALL");
    }
    
    // Handle other errors
    const errorText = await response.text().catch(() => "");
    
    // Check for specific error messages
    if (errorText.includes("Google authentication")) {
      throw new Error("Google authentication failed. Please try signing in again.");
    }
    
    if (errorText.includes("Google Drive")) {
      throw new Error("Failed to save to Google Drive. Please check your Google account permissions.");
    }
    
    // Generic error fallback
    throw new Error(errorText || "Failed to save to Google Docs");
  }

  // Parse successful response
  const result: GoogleDocExportResponse = await response.json();
  
  // Validate response structure
  if (!result.fileId || !result.fileUrl) {
    throw new Error("Invalid response from Google Docs export service");
  }
  
  return result;
}

/**
 * Open Google Doc in new tab
 * @param fileUrl - Google Docs file URL
 */
export function openGoogleDoc(fileUrl: string): void {
  window.open(fileUrl, '_blank', 'noopener,noreferrer');
}

/**
 * Generate filename for Google Doc
 * @param businessName - Business name
 * @param mode - Plan mode (lean or full)
 * @returns string Formatted filename
 */
export function generateGoogleDocFilename(businessName: string, mode: string): string {
  const sanitizedName = businessName.replace(/[^a-zA-Z0-9\s-]/g, '').trim();
  const modeLabel = mode === 'full' ? 'Full' : 'Lean';
  return `${sanitizedName} - ${modeLabel} Business Plan`;
}