/**
 * Frontend service for business plan export functionality
 * Handles PDF and DOCX export downloads
 */

import { PlanResponse } from "./plan";
import { auth } from "@/lib/firebase";

/**
 * Export a business plan to PDF or DOCX format
 * @param plan - The generated business plan to export
 * @param format - The export format (pdf or docx)
 * @param businessName - Optional business name for filename
 * @throws Error with "PAYWALL" message for paywall errors
 */
export async function exportPlan(
  plan: PlanResponse, 
  format: "pdf" | "docx", 
  businessName?: string
): Promise<void> {
  // Get Firebase ID token for authentication
  const user = auth.currentUser;
  const token = user ? await user.getIdToken() : null;
  
  // Prepare headers with authentication
  const headers: Record<string, string> = {
    "Content-Type": "application/json"
  };
  
  // Add authentication token if available
  if (token) {
    headers["Authorization"] = `Bearer ${token}`;
  }
  
  // Add development bypass header in development mode
  const isDevelopment = import.meta.env.MODE === 'development';
  if (isDevelopment) {
    headers["x-dev-bypass"] = "development";
  }

  const response = await fetch("/api/ai/plan/export", {
    method: "POST",
    headers,
    body: JSON.stringify({ plan, format, businessName }),
  });

  if (!response.ok) {
    if (response.status === 402) {
      throw new Error("PAYWALL");
    }
    const errorText = await response.text().catch(() => "");
    throw new Error(errorText || "Export failed");
  }

  // Handle the file download
  const blob = await response.blob();
  const filename = `${(businessName || "Business")}-Business-Plan.${format}`;
  const url = window.URL.createObjectURL(blob);
  const link = document.createElement("a");
  link.href = url;
  link.download = filename;
  document.body.appendChild(link);
  link.click();
  link.remove();
  window.URL.revokeObjectURL(url);
}