/**
 * Frontend service client for business plan generation
 * Connects to the backend AI plan generation endpoint
 */

export type PlanMode = "lean" | "full";
export type PlanTone = "Professional" | "Friendly" | "Bold" | "Minimal";

export type PlanRequest = {
  mode: PlanMode;
  businessName: string;
  description: string;
  industry?: string;
  audience?: string;
  problem?: string;
  solution?: string;
  advantage?: string;
  pricing?: string;
  timeframeMonths?: 3 | 6 | 12;
  tone?: PlanTone;
};

export interface PlanSection {
  id: string;
  title: string;
  content: string;
}

export type PlanResponse = {
  mode: PlanMode;
  sections: PlanSection[];
  finance?: {
    assumptions: string[];
    monthly_table?: Array<{ month: string; revenue: number; costs: number; profit: number }>;
  };
};

/**
 * Generate a business plan using AI
 * @param req - The business plan request parameters
 * @returns Promise containing the generated business plan
 */
export async function generatePlan(req: PlanRequest): Promise<PlanResponse> {
  const response = await fetch('/api/ai/plan', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
    },
    credentials: 'include', // Include authentication cookies
    body: JSON.stringify(req),
  });

  if (!response.ok) {
    const errorText = await response.text();
    throw new Error(`Failed to generate business plan: ${response.status} ${errorText}`);
  }

  return response.json();
}