// src/services/ai/business-names.ts
export type BusinessNameStyle = 'Modern' | 'Classic' | 'Creative' | 'Professional' | 'Playful';

export interface GenerateBusinessNamesInput {
  industry: string;
  keywords: string;
  style?: BusinessNameStyle;
  count?: number;

  // Creativity control (0-100)
  creativity?: number; // replaces randomness, 60 = default balanced
  description?: string;
  vibe?: string; // single vibe selection
  length_range?: [number, number];
  starts_with?: string;
  must_include?: string; // comma sep ok
  must_exclude?: string; // comma sep ok
}

export interface GeneratedBusinessName {
  name: string;
}

export async function generateBusinessNames(
  input: GenerateBusinessNamesInput
): Promise<GeneratedBusinessName[]> {
  const body = {
    ...input,
    count: input.count ?? 12,
  };

  try {
    const res = await fetch('/api/business-names/generate', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(body),
    });
    if (res.ok) {
      const data = await res.json();
      // Expect: { names: string[] }
      if (Array.isArray(data?.names)) {
        return data.names.map((name: string) => ({ name }));
      }
    }
  } catch (err) {
    console.warn('Falling back to local generator:', err);
  }

  // Fallback local generator (so the UI still works in dev)
  const seeds = synthesize(body);
  return seeds.map(name => ({ name }));
}

// --- helpers -------------------------------------------------------------

function synthesize(input: GenerateBusinessNamesInput): string[] {
  const base = input.keywords
    .split(',')
    .map(s => s.trim())
    .filter(Boolean);

  // a tiny brandable mixer
  const endings = ['ly', 'io', 'ify', 'verse', 'lab', 'nest', 'mint', 'nova', 'hub', 'loop', 'grid', 'forge'];
  const results = new Set<string>();

  for (const k of base) {
    const K = k.replace(/[^a-z0-9]/gi, '');
    if (!K) continue;

    results.add(cap(K));
    results.add(`${cap(K)}Lab`);
    results.add(`${cap(K)}Nova`);
    results.add(`${cap(K)}ly`);
    results.add(`${cap(K)}ify`);

    for (const e of endings) results.add(cap(K) + e);

    // mix with industry token
    const firstIndustry = input.industry.split(/[,\s]/)[0]?.replace(/[^a-z0-9]/gi, '') || '';
    if (firstIndustry) {
      results.add(cap(firstIndustry) + cap(K));
      results.add(cap(K) + cap(firstIndustry));
    }
  }

  // pad if too short
  while (results.size < (input.count ?? 12)) {
    const r = randomToken();
    results.add(r);
  }

  // Apply filters
  let filteredResults = Array.from(results);

  // Filter by must_exclude (words that should NOT be in the name)
  if (input.must_exclude?.trim()) {
    const excludeWords = input.must_exclude.split(',').map(w => w.trim().toLowerCase()).filter(Boolean);
    filteredResults = filteredResults.filter(name => {
      const nameLower = name.toLowerCase();
      return !excludeWords.some(word => nameLower.includes(word));
    });
  }

  // Filter by must_include (words that MUST be in the name)
  if (input.must_include?.trim()) {
    const includeWords = input.must_include.split(',').map(w => w.trim().toLowerCase()).filter(Boolean);
    filteredResults = filteredResults.filter(name => {
      const nameLower = name.toLowerCase();
      return includeWords.some(word => nameLower.includes(word));
    });
  }

  // Filter by starts_with
  if (input.starts_with?.trim()) {
    const startsWith = input.starts_with.trim().toLowerCase();
    filteredResults = filteredResults.filter(name => 
      name.toLowerCase().startsWith(startsWith)
    );
  }

  // Filter by length_range
  if (input.length_range) {
    const [minLen, maxLen] = input.length_range;
    filteredResults = filteredResults.filter(name => 
      name.length >= minLen && name.length <= maxLen
    );
  }

  // If filtering removed too many results, generate more and refilter
  if (filteredResults.length < (input.count ?? 12)) {
    // Generate more backup names
    const extraNames = [];
    for (let i = 0; i < 20; i++) {
      extraNames.push(randomToken());
    }
    
    // Apply same filters to backup names
    let filteredExtras = extraNames;
    
    if (input.must_exclude?.trim()) {
      const excludeWords = input.must_exclude.split(',').map(w => w.trim().toLowerCase()).filter(Boolean);
      filteredExtras = filteredExtras.filter(name => {
        const nameLower = name.toLowerCase();
        return !excludeWords.some(word => nameLower.includes(word));
      });
    }
    
    if (input.must_include?.trim()) {
      const includeWords = input.must_include.split(',').map(w => w.trim().toLowerCase()).filter(Boolean);
      filteredExtras = filteredExtras.filter(name => {
        const nameLower = name.toLowerCase();
        return includeWords.some(word => nameLower.includes(word));
      });
    }
    
    if (input.starts_with?.trim()) {
      const startsWith = input.starts_with.trim().toLowerCase();
      filteredExtras = filteredExtras.filter(name => 
        name.toLowerCase().startsWith(startsWith)
      );
    }
    
    if (input.length_range) {
      const [minLen, maxLen] = input.length_range;
      filteredExtras = filteredExtras.filter(name => 
        name.length >= minLen && name.length <= maxLen
      );
    }
    
    filteredResults = [...filteredResults, ...filteredExtras];
  }

  return filteredResults.slice(0, input.count ?? 12);
}

function cap(s: string) {
  return s.charAt(0).toUpperCase() + s.slice(1).toLowerCase();
}

function randomToken() {
  const pieces = ['Caffe', 'Mocha', 'Brew', 'Pulse', 'Prime', 'Astra', 'Neo', 'Cortex', 'Nova', 'Mint', 'Nest', 'Foundry'];
  const a = pieces[Math.floor(Math.random() * pieces.length)];
  const b = pieces[Math.floor(Math.random() * pieces.length)];
  return a === b ? `${a}` : `${a}${b}`;
}