import { BrandKit, BrandPersonality, Palette, LogoVariant } from "@/types/brand";
import { adjust, rotateHue, textOn } from "@/utils/colors";
import { pickFontPair } from "@/utils/fonts";

// Seed palettes by personality (you can tune these later)
const BASE: Record<BrandPersonality, string> = {
  Modern:   "#4274b9", // your blue
  Natural:  "#6fc282", // light green
  Luxury:   "#231f20", // charcoal
  Friendly: "#FF8800", // orange
};

function buildPalette(personality: BrandPersonality): Palette {
  const primary = BASE[personality];
  const secondary =
    personality === "Luxury" ? rotateHue(primary, 20) : rotateHue(primary, 150);
  const accent =
    personality === "Natural" ? rotateHue(primary, -40) : rotateHue(primary, -30);
  const neutral = personality === "Luxury" ? "#30342d" : "#f3f4f6";
  const surface = personality === "Luxury" ? "#1f1f21" : "#ffffff";
  const textPrimary = personality === "Luxury" ? "#f5f5f5" : "#1f2937";
  const textOnPrimary = textOn(primary);

  // a few shades for UI
  const shades: Record<string, string> = {
    100: adjust(primary, { dl: 0.35 }),
    200: adjust(primary, { dl: 0.25 }),
    300: adjust(primary, { dl: 0.15 }),
    400: adjust(primary, { dl: 0.08 }),
    500: primary,
    600: adjust(primary, { dl: -0.06 }),
    700: adjust(primary, { dl: -0.12 }),
  };

  return {
    name: personality,
    primary, secondary, accent, neutral, surface, textPrimary, textOnPrimary, shades
  };
}

function mkLogoSVG({
  text,
  color,
  bg = "none",
  ring = false,
  font = "Barlow Condensed",
}: { text: string; color: string; bg?: string; ring?: boolean; font?: string; }): string {
  const initial = (text?.trim()?.[0] || "B").toUpperCase();
  const textColor = textOn(color);
  const markOnly = `
    <svg xmlns="http://www.w3.org/2000/svg" width="640" height="640" viewBox="0 0 640 640">
      <rect width="640" height="640" rx="96" fill="${bg}" />
      <circle cx="320" cy="320" r="148" fill="${color}" />
      ${ring ? `<circle cx="320" cy="320" r="208" fill="none" stroke="${color}" stroke-width="20" opacity="0.5" />` : ""}
      <text x="50%" y="54%" text-anchor="middle" font-family="${font}" font-size="220" font-weight="700" fill="${textColor}">${initial}</text>
    </svg>`.trim();

  const fullLockup = `
    <svg xmlns="http://www.w3.org/2000/svg" width="1200" height="400" viewBox="0 0 1200 400">
      <rect width="1200" height="400" rx="32" fill="${bg}" />
      <g transform="translate(40,40)">
        <circle cx="160" cy="160" r="96" fill="${color}" />
        ${ring ? `<circle cx="160" cy="160" r="140" fill="none" stroke="${color}" stroke-width="14" opacity="0.5" />` : ""}
        <text x="160" y="180" text-anchor="middle" font-family="${font}" font-size="120" font-weight="700" fill="${textOn(color)}">${initial}</text>
      </g>
      <text x="360" y="190" font-family="${font}" font-size="120" font-weight="700" fill="${color}">${text}</text>
      <text x="360" y="250" font-family="Inter" font-size="28" fill="#6b7280">Make Your First Impression Your Best!</text>
    </svg>`.trim();

  return ring ? fullLockup : markOnly;
}

function logoVariants(name: string, palette: Palette, headingFont: string): LogoVariant[] {
  return [
    { filename: "logo-primary.svg", svg: mkLogoSVG({ text: name, color: palette.primary, bg: "none", ring: true, font: headingFont }) },
    { filename: "logo-reverse.svg", svg: mkLogoSVG({ text: name, color: palette.primary, bg: palette.primary, ring: false, font: headingFont }) },
    { filename: "logo-mark.svg", svg: mkLogoSVG({ text: name, color: palette.accent, bg: "none", ring: false, font: headingFont }) },
  ];
}

export function generateBrandKit({
  businessName,
  industry,
  personality,
}: {
  businessName: string;
  industry?: string;
  personality: BrandPersonality;
}): BrandKit {
  const palette = buildPalette(personality);
  const fonts = pickFontPair(personality);
  const logos = logoVariants(businessName || "Your Brand", palette, fonts.heading);

  return {
    businessName: businessName || "Your Brand",
    industry,
    personality,
    palette,
    fonts,
    logos,
    createdAt: Date.now(),
  };
}