/**
 * AI Client Service for Business Plan Section Writing Assistant
 * Handles communication with the AI section generation endpoint
 */

import type { AiJob, AiResponse, AiError } from "@shared/ai-types";
import { auth } from "@/lib/firebase";

// Custom error class for AI-specific errors
export class AiClientError extends Error {
  constructor(
    public code: AiError,
    message: string,
    public statusCode?: number
  ) {
    super(message);
    this.name = 'AiClientError';
  }
}

/**
 * Main AI client service for section content operations
 */
export class AiClientService {
  private baseUrl = '/api/ai';

  /**
   * Process an AI job for section content generation/modification
   * @param job - The AI job specification
   * @returns Promise containing the AI response
   */
  async processJob(job: AiJob): Promise<AiResponse> {
    try {
      const authHeader = await this.getAuthHeader();
      const response = await fetch(`${this.baseUrl}/section`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          ...authHeader,
        },
        credentials: 'include', // Include cookies for authentication
        body: JSON.stringify(job),
      });

      // Handle HTTP errors
      if (!response.ok) {
        return this.handleHttpError(response);
      }

      const data = await response.json();
      
      // Validate response structure
      if (!this.isValidAiResponse(data)) {
        throw new AiClientError(
          'invalid_request',
          'Invalid response format from AI service'
        );
      }

      return data;

    } catch (error) {
      // Handle network errors
      if (error instanceof AiClientError) {
        throw error;
      }

      if (error instanceof TypeError && error.message.includes('fetch')) {
        throw new AiClientError(
          'service_unavailable',
          'Unable to connect to AI service. Please check your connection.'
        );
      }

      // Generic error fallback
      throw new AiClientError(
        'unknown_error',
        error instanceof Error ? error.message : 'Unknown error occurred'
      );
    }
  }

  /**
   * Generate new content for a section
   */
  async generateContent(
    sectionKind: string,
    sectionTitle: string,
    tone: string,
    context?: AiJob['context'],
    existingContent?: string,
    length?: string,
    userPrompt?: string
  ): Promise<AiResponse> {
    return this.processJob({
      action: 'generate',
      tone: tone as any, // Type assertion for ToneOption
      length: length as any, // Type assertion for LengthOption
      sectionKind,
      sectionTitle,
      context,
      existingContent,
      userPrompt,
    });
  }

  /**
   * Rephrase existing content in a different tone
   */
  async rephraseContent(
    sectionKind: string,
    sectionTitle: string,
    existingContent: string,
    tone: string,
    context?: AiJob['context'],
    userPrompt?: string
  ): Promise<AiResponse> {
    return this.processJob({
      action: 'rephrase',
      tone: tone as any,
      sectionKind,
      sectionTitle,
      existingContent,
      context,
      userPrompt,
    });
  }

  /**
   * Expand existing content with more details
   */
  async expandContent(
    sectionKind: string,
    sectionTitle: string,
    existingContent: string,
    tone: string,
    context?: AiJob['context'],
    userPrompt?: string
  ): Promise<AiResponse> {
    return this.processJob({
      action: 'expand',
      tone: tone as any,
      sectionKind,
      sectionTitle,
      existingContent,
      context,
      userPrompt,
    });
  }

  /**
   * Summarize existing content to make it more concise
   */
  async summarizeContent(
    sectionKind: string,
    sectionTitle: string,
    existingContent: string,
    tone: string,
    context?: AiJob['context'],
    userPrompt?: string
  ): Promise<AiResponse> {
    return this.processJob({
      action: 'summarize',
      tone: tone as any,
      sectionKind,
      sectionTitle,
      existingContent,
      context,
      userPrompt,
    });
  }

  /**
   * Generate structured template data with AI
   */
  async generateStructuredTemplate(
    templateKey: string,
    businessBrief?: any
  ): Promise<{ data: Record<string, string>, isSeedDraft?: boolean }> {
    const authHeader = await this.getAuthHeader();
    const response = await fetch(`${this.baseUrl}/structured-template`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        ...authHeader,
      },
      credentials: 'include',
      body: JSON.stringify({ templateKey, businessBrief }),
    });

    if (!response.ok) {
      // Special case for authentication errors
      if (response.status === 401) {
        throw new Error('Authentication required. Please log in to use AI features.');
      }
      
      const errorText = await response.text().catch(() => response.statusText);
      throw new Error(`Failed to generate template: ${errorText}`);
    }

    const result = await response.json();
    if (!result.success || !result.data) {
      throw new Error('Invalid response from template generation service');
    }
    
    return { data: result.data, isSeedDraft: result.isSeedDraft };
  }

  /**
   * Get authentication header with Firebase token
   */
  private async getAuthHeader(): Promise<Record<string, string>> {
    try {
      const currentUser = auth.currentUser;
      if (!currentUser) {
        console.warn('No authenticated user found for AI request');
        return {};
      }
      
      const token = await currentUser.getIdToken();
      return {
        'Authorization': `Bearer ${token}`
      };
    } catch (error) {
      console.error('Failed to get Firebase auth token:', error);
      return {};
    }
  }

  /**
   * Handle HTTP error responses
   */
  private async handleHttpError(response: Response): Promise<AiResponse> {
    const statusCode = response.status;
    let errorMessage: string;
    let errorCode: AiError;

    try {
      const errorData = await response.json();
      errorMessage = errorData.error || errorData.message || `HTTP ${statusCode}`;
    } catch {
      errorMessage = `HTTP ${statusCode}: ${response.statusText}`;
    }

    // Map HTTP status codes to AI error types
    switch (statusCode) {
      case 401:
        errorCode = 'api_key_missing';
        errorMessage = 'Authentication required. Please log in.';
        break;
      case 403:
        errorCode = 'api_key_missing';
        errorMessage = 'Access denied. Please check your subscription.';
        break;
      case 429:
        errorCode = 'rate_limit_exceeded';
        errorMessage = 'Rate limit exceeded. Please try again later.';
        break;
      case 400:
        errorCode = 'invalid_request';
        break;
      case 503:
      case 502:
      case 504:
        errorCode = 'service_unavailable';
        errorMessage = 'AI service temporarily unavailable. Please try again.';
        break;
      default:
        errorCode = 'unknown_error';
    }

    return {
      success: false,
      error: errorMessage,
    };
  }

  /**
   * Validate AI response structure
   */
  private isValidAiResponse(data: any): data is AiResponse {
    return (
      typeof data === 'object' &&
      data !== null &&
      typeof data.success === 'boolean' &&
      (data.success === false || typeof data.content === 'string')
    );
  }
}

// Export singleton instance
export const aiClient = new AiClientService();

// Convenience functions for common operations
export const generateSectionContent = aiClient.generateContent.bind(aiClient);
export const rephraseSectionContent = aiClient.rephraseContent.bind(aiClient);
export const expandSectionContent = aiClient.expandContent.bind(aiClient);
export const summarizeSectionContent = aiClient.summarizeContent.bind(aiClient);