import React from "react";
import TemplatePage from "@/components/TemplatePage";
import { Button } from "@/components/ui/button";
import { Link } from "wouter";
import priceImage from "@/assets/price.png"; // reuse banner art for consistency

export default function WebAddOns() {
  const Check = (props: React.SVGProps<SVGSVGElement>) => (
    <svg
      viewBox="0 0 24 24"
      fill="none"
      aria-hidden="true"
      {...props}
      className={"w-5 h-5 " + (props.className ?? "")}
    >
      <path
        d="M20 6L9 17l-5-5"
        stroke="currentColor"
        strokeWidth="2"
        strokeLinecap="round"
        strokeLinejoin="round"
      />
    </svg>
  );

  // Edit these slugs if your WHMCS store category URLs differ
  const STORE = "https://billing.ibrandbiz.com/index.php?rp=/store";
  const addOnLinks = {
    ssl_basic: `${STORE}/ssl-certificates`,
    ssl_premium: `${STORE}/ssl-certificates`, // EV/Wildcard typically same category
    backups: `${STORE}/website-backup`,
    dedicated_ip: `${STORE}/dedicated-ip`,
    storage: `${STORE}`, // fallback to store root if no specific category
    priority_support: `${STORE}`,
    email_security: `${STORE}/email-security`,
    sitelock: `${STORE}/sitelock`,
    spam_virus: `${STORE}/email-security`,
    builder: `${STORE}/weebly`,
  };

  const Card = ({
    title,
    blurb,
    bullets,
    cta,
    href,
    price,
  }: {
    title: string;
    blurb: string;
    bullets: string[];
    cta: string;
    href: string;
    price: string;
  }) => (
    <div className="bg-white rounded-xl border shadow-sm p-6 flex flex-col">
      <h3 className="text-xl font-bold mb-2">{title}</h3>
      <div className="text-2xl font-extrabold text-primary mb-3" aria-label={`Price: ${price}`}>
        {price}
      </div>
      <p className="text-gray-600 mb-4">{blurb}</p>
      <ul className="space-y-2 mb-6 flex-grow">
        {bullets.map((b, i) => (
          <li key={i} className="flex items-start gap-2">
            <Check className="text-green-500 mt-0.5" />
            <span>{b}</span>
          </li>
        ))}
      </ul>
      <a href={href} target="_blank" rel="noopener noreferrer" className="w-full">
        <Button className="w-full bg-primary hover:bg-primary/90 text-white">
          {cta}
        </Button>
      </a>
      <p className="text-xs text-gray-500 text-center mt-2">
        Selected during hosting checkout or from your client area.
      </p>
    </div>
  );

  return (
    <TemplatePage
      bannerTitle="Web Hosting Add-Ons"
      bannerSubtitle="Upgrade performance, security, and workflow — only what you need, when you need it."
      bannerImage={priceImage}
      flipImage={false}
    >
      <div className="w-full">
        <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
          {/* Intro */}
          <div className="text-center max-w-3xl mx-auto mb-10">
            <p className="text-gray-600">
              Power up your hosting with focused upgrades. Mix and match security,
              storage, backups, email protection, and tools — all managed in your
              client portal.
            </p>
          </div>

          {/* Grid */}
          <div className="grid gap-8 md:grid-cols-2 lg:grid-cols-3">
            <Card
              title="Basic SSL Certificate"
              blurb="Lock your site with HTTPS and the padlock icon for instant trust."
              bullets={["Domain Validation (DV)", "Auto renewals available", "SEO and browser trust"]}
              cta="Add SSL"
              href={addOnLinks.ssl_basic}
              price="$9.99/year"
            />

            <Card
              title="Premium SSL (Wildcard/EV)"
              blurb="Max credibility with advanced certificate options for brands & stores."
              bullets={["Wildcard & EV options", "Stronger identity validation", "Ideal for ecommerce"]}
              cta="Explore Premium SSL"
              href={addOnLinks.ssl_premium}
              price="$29.99/year"
            />

            <Card
              title="Daily Backups + Restore"
              blurb="Automated daily snapshots and one-click restores when you need it."
              bullets={["Daily offsite backups", "Point-in-time restore", "Easy rollback safety net"]}
              cta="Add Backups"
              href={addOnLinks.backups}
              price="$2.99/month"
            />

            <Card
              title="Dedicated IP Address"
              blurb="Improve email deliverability and enable certain premium SSL needs."
              bullets={["Better email reputation", "Required for some SSLs", "Clean, stable IP"]}
              cta="Get Dedicated IP"
              href={addOnLinks.dedicated_ip}
              price="$2.49/month"
            />

            <Card
              title="Extra SSD Storage"
              blurb="Add fast storage packs as your media library or traffic grows."
              bullets={["+25–100 GB packs", "NVMe/SSD speed", "Scale as needed"]}
              cta="Add Storage"
              href={addOnLinks.storage}
              price="$2.99/month"
            />

            <Card
              title="Priority 24/7 Support Add-on"
              blurb="Jump the queue with SLA-priority assistance for mission-critical ops."
              bullets={["SLA response targets", "Faster resolutions", "Proactive guidance"]}
              cta="Upgrade Support"
              href={addOnLinks.priority_support}
              price="$3.49/month"
            />

            <Card
              title="Professional Email (per domain)"
              blurb="Create professional addresses and route mail with filtering protection."
              bullets={["Spam & virus filtering", "DKIM/SPF guidance", "Safer brand email"]}
              cta="Secure My Email"
              href={addOnLinks.email_security}
              price="$1.99/month"
            />

            <Card
              title="SiteLock Security (Malware Scan)"
              blurb="Automated daily scans and malware removal to keep you clean."
              bullets={["Daily malware scans", "Blacklist monitoring", "Auto remove options"]}
              cta="Add SiteLock"
              href={addOnLinks.sitelock}
              price="$3.99/month"
            />

            <Card
              title="Spam & Virus Protection (Email Security)"
              blurb="Harden mailboxes with advanced filters and quarantine tools."
              bullets={["Advanced heuristics", "Sender reputation checks", "Admin quarantine"]}
              cta="Activate Protection"
              href={addOnLinks.spam_virus}
              price="$1.99/month"
            />

            <Card
              title="Weebly Site Builder (Drag & Drop)"
              blurb="Drag-and-drop builder for quick launches and simple edits."
              bullets={["No-code editing", "Mobile-ready themes", "Fast page creation"]}
              cta="Add Builder"
              href={addOnLinks.builder}
              price="$4.99/month"
            />
          </div>

          {/* Back link */}
          <div className="text-center mt-12">
            <Link href="/pricing/web-services">
              <Button className="bg-primary hover:bg-primary/90 text-white">
                ← Back to Hosting Plans
              </Button>
            </Link>
          </div>

          {/* Note */}
          <div className="mt-8 text-center text-sm text-gray-500">
            Tip: If any link 404s, adjust the category slug in the{" "}
            <code className="bg-gray-100 px-1 py-0.5 rounded">addOnLinks</code> map at the top.
          </div>
        </div>
      </div>
    </TemplatePage>
  );
}