import React, { useMemo, useState } from "react";
import { useAuth } from "@/contexts/AuthContext";
import { useLocation } from "wouter";
import { useToast } from "@/hooks/use-toast";
import TemplatePage from "@/components/TemplatePage";
import SubscribeModal from "@/components/billing/SubscribeModal";
import priceImage from "@/assets/price.png";
import { Button } from "@/components/ui/button";

/**
 * Business Development Pricing Page
 * - 3 cards: Free, Pro (Most Popular), Enterprise
 * - Billing toggle: Monthly / Yearly
 * - Pro: $19/mo OR $190/yr (no discount badge/copy)
 * - Accessible button-group toggle
 */
export default function BusinessDevelopmentPricing() {
  const { currentUser } = useAuth();
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const [billing, setBilling] = useState<"monthly" | "yearly">("monthly");
  const [isModalOpen, setIsModalOpen] = useState(false);

  const prices = useMemo(
    () => ({
      free: { amount: "$0", unit: "/forever" },
      pro:
        billing === "monthly"
          ? { amount: "$19", unit: "/month" }
          : { amount: "$190", unit: "/year" },
      enterprise: { amount: "Custom", unit: "/contact us" },
    }),
    [billing]
  );

  // Handle Pro button click - show payment modal
  const handleProUpgrade = () => {
    if (!currentUser) {
      // Not signed in - redirect to signup page with pro intent
      setLocation(`/signup?intent=pro&billing=${billing}`);
      return;
    }

    // Show payment modal
    setIsModalOpen(true);
  };

  const Check = (props: React.SVGProps<SVGSVGElement>) => (
    <svg
      viewBox="0 0 24 24"
      fill="none"
      aria-hidden="true"
      {...props}
      className={"w-5 h-5 " + (props.className ?? "")}
    >
      <path d="M20 6L9 17l-5-5" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
    </svg>
  );

  return (
    <TemplatePage 
      bannerTitle="Business Development Pricing" 
      bannerSubtitle="Choose the perfect plan for your business development needs"
      bannerImage={priceImage}
      flipImage={true}
    >
      <div className="w-full">
      {/* Content */}
      <div className="max-w-5xl mx-auto px-4 sm:px-6 lg:px-8 py-8 text-center">

        {/* Billing Toggle */}
        <div className="mt-6 flex justify-center" role="group" aria-label="Billing cycle">
          <div className="inline-flex items-center rounded-full bg-gray-100 p-1">
            <button
              type="button"
              onClick={() => setBilling("monthly")}
              aria-pressed={billing === "monthly"}
              data-billing="monthly"
              className={
                "px-4 py-2 rounded-full text-sm font-medium transition " +
                (billing === "monthly" ? "bg-green-500 text-white shadow" : "text-gray-700")
              }
            >
              Monthly
            </button>
            <button
              type="button"
              onClick={() => setBilling("yearly")}
              aria-pressed={billing === "yearly"}
              data-billing="yearly"
              className={
                "px-4 py-2 rounded-full text-sm font-medium transition " +
                (billing === "yearly" ? "bg-green-500 text-white shadow" : "text-gray-700")
              }
            >
              Yearly
            </button>
          </div>
        </div>
      </div>

      {/* Cards */}
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-10 grid gap-6 md:grid-cols-2 justify-center">
        {/* Free */}
        <div className="rounded-xl border bg-white shadow-sm p-6 flex flex-col">
          <h3 className="text-xl font-bold text-center">Free</h3>
          <div className="mt-3 text-center">
            <Price amount={prices.free.amount} unit={prices.free.unit} />
          </div>

          <p className="mt-4 text-center text-gray-600">
            Perfect for getting started with basic brand tools
          </p>

          <ul className="mt-6 space-y-3">
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>Brand Name Wizard — <strong>3</strong> name results per run</span>
            </li>
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>Slogan Generator — <strong>3</strong> slogan results per run</span>
            </li>
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>Basic Logo Composer — edit 1 AI draft (no Brand Kit save)</span>
            </li>
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>Standard support</span>
            </li>
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>Community access</span>
            </li>
          </ul>

          <a
            href="/signup?intent=free"
            className="mt-6 w-full inline-flex justify-center items-center rounded-lg border px-4 py-2 font-medium"
            data-testid="button-get-started-free"
          >
            Get Started Free
          </a>
        </div>

        {/* Pro (Most Popular) */}
        <div className="relative rounded-xl border-2 border-green-500 bg-white shadow-lg p-6 flex flex-col">
          {/* Badge */}
          <div className="absolute -top-3 inset-x-0 flex justify-center">
            <div className="px-3 py-1 rounded-full bg-green-500 text-white text-sm font-semibold shadow">
              ★ Most Popular
            </div>
          </div>

          <h3 className="text-xl font-bold text-center mt-2">Pro</h3>

          <div className="mt-3 text-center">
            <Price amount={prices.pro.amount} unit={prices.pro.unit} fadeKey={billing} />
          </div>

          <p className="mt-4 text-center text-gray-600">
            Everything you need to build a complete professional brand
          </p>

          <ul className="mt-6 space-y-3">
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>Brand Name Wizard — <strong>10</strong> results/run + tone/style filters</span>
            </li>
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>Slogan Generator — <strong>10</strong> results/run + tone/style filters</span>
            </li>
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>AI Logo Creator — up to <strong>8 AI options</strong>, open in Composer</span>
            </li>
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>Brand Kit — full kit creation (logos, colors, fonts, slogans)</span>
            </li>
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>AI Business Plan Developer</span>
            </li>
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>Domain search integration</span>
            </li>
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>Export — PDF, DOCX, Google Docs</span>
            </li>
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>Premium logo composers — templates + advanced tools</span>
            </li>
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>Priority support</span>
            </li>
            <li className="flex items-start gap-2">
              <Check className="text-green-500 mt-0.5" />
              <span>Commercial usage rights</span>
            </li>
          </ul>

          <button
            onClick={handleProUpgrade}
            className="mt-6 w-full inline-flex justify-center items-center rounded-lg bg-green-500 text-white px-4 py-2 font-semibold hover:opacity-95"
            data-testid="button-upgrade-pro"
          >
            Upgrade to Pro
          </button>
        </div>
      </div>

      {/* Payment Modal */}
      <SubscribeModal
        open={isModalOpen}
        onOpenChange={setIsModalOpen}
        priceId={billing}
        planLabel={billing === 'monthly' ? '$19/month' : '$190/year'}
        userEmail={currentUser?.email}
      />
      </div>
    </TemplatePage>
  );
}

/** Price component with a tiny fade on change (CSS-only, no lib). */
function Price({
  amount,
  unit,
  fadeKey,
}: {
  amount: string;
  unit: string;
  fadeKey?: string | number;
}) {
  return (
    <div className="inline-flex items-baseline gap-2">
      <span
        key={String(fadeKey) + amount}
        className="text-4xl md:text-5xl font-extrabold text-green-500 animate-[fade_200ms_ease]"
      >
        {amount}
      </span>
      <span className="text-gray-600">{unit}</span>
      {/* local keyframes */}
      <style>{`@keyframes fade{0%{opacity:.2;transform:translateY(4px)}100%{opacity:1;transform:translateY(0)}}`}</style>
    </div>
  );
}