import { useEffect, useMemo, useState } from "react";
import { useLocation, useParams, Link } from "wouter";
import { Loader2 } from "lucide-react";

// If you already have your real customizer, import it here:
// import { LogoTemplateBuilder } from "@/pages/logo-templates/LogoTemplateBuilder";

type StatePayload = {
  templateId: string;
  name?: string;
  svgUrl?: string;
  rasterUrl?: string;
  defaults?: Record<string, string>;
};

export default function LogoTemplateDetail() {
  const { id = "" } = useParams<{ id: string }>();
  const [location] = useLocation();
  
  // Get initial data from sessionStorage (our Wouter equivalent of location.state)
  const getInitialState = (): Partial<StatePayload> => {
    try {
      const storedData = sessionStorage.getItem('templateData');
      if (storedData) {
        const parsed = JSON.parse(storedData);
        if (parsed.templateId === id) {
          return parsed;
        }
      }
    } catch (error) {
      console.warn("Failed to parse stored template data:", error);
    }
    return {};
  };

  const initial = getInitialState();

  const [loading, setLoading] = useState(!initial.svgUrl && !initial.rasterUrl);
  const [data, setData] = useState<StatePayload>({
    templateId: initial.templateId || id,
    name: initial.name,
    svgUrl: initial.svgUrl,
    rasterUrl: initial.rasterUrl,
    defaults: initial.defaults,
  });

  // If we didn't receive the assets via state, fetch by id from backend API
  useEffect(() => {
    if (data.svgUrl || data.rasterUrl) return; // already have it from state
    (async () => {
      try {
        setLoading(true);
        
        // Use backend API instead of direct Firestore access
        const response = await fetch(`/api/logo-templates-firebase/${id}`);
        if (!response.ok) throw new Error(`HTTP ${response.status}`);
        const template = await response.json();
        
        if (template) {
          setData(d => ({
            ...d,
            name: template.name,
            svgUrl: template.assets?.svgDownloadUrl,
            rasterUrl: template.assets?.previewDownloadUrl,
            defaults: template.defaults || {},
          }));
        }
      } catch (err) {
        console.error("Failed to load template:", err);
      } finally {
        setLoading(false);
      }
    })();
  }, [id, data.svgUrl, data.rasterUrl]);

  // Once props are known, map to your customizer props here
  const customizerProps = useMemo(() => ({
    templateId: data.templateId,
    name: data.name || data.templateId,
    backgroundSvgUrl: data.svgUrl,          // <— background "logo behind text"
    backgroundRasterUrl: data.rasterUrl,    // fallback
    defaults: data.defaults || {},          // { title, tagline, color, etc. }
  }), [data]);

  return (
    <div className="p-6 space-y-4">
      <Link href="/business-development/ai-logo-creator" className="text-sm text-muted-foreground hover:underline">
        ← Back to AI Logo Creator
      </Link>

      <h1 className="text-2xl font-semibold">{customizerProps.name}</h1>

      {loading ? (
        <div className="flex items-center gap-2 text-muted-foreground">
          <Loader2 className="w-4 h-4 animate-spin" /> Loading template…
        </div>
      ) : !customizerProps.backgroundSvgUrl && !customizerProps.backgroundRasterUrl ? (
        <p className="text-red-500">Template assets not found.</p>
      ) : (
        <>
          {/* If you already have your real customizer, render it here and pass props: */}
          {/* <LogoTemplateBuilder {...customizerProps} /> */}

          {/* TEMPORARY PREVIEW so QA can confirm asset + layering works */}
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
            <div className="relative border rounded-lg bg-white p-8 min-h-[420px] overflow-hidden">
              {customizerProps.backgroundSvgUrl ? (
                <img
                  src={customizerProps.backgroundSvgUrl}
                  alt=""
                  className="absolute inset-0 w-full h-full object-contain opacity-90 pointer-events-none"
                />
              ) : (
                <img
                  src={customizerProps.backgroundRasterUrl!}
                  alt=""
                  className="absolute inset-0 w-full h-full object-contain opacity-90 pointer-events-none"
                />
              )}

              {/* "Text that moves/curves" placeholder — your builder will control these */}
              <div className="relative z-10 flex flex-col items-center justify-center h-full">
                <h2 className="text-4xl font-bold tracking-tight">
                  {customizerProps.defaults?.title || customizerProps.defaults?.brandName || customizerProps.defaults?.Brand_Name || "Your Brand"}
                </h2>
                <p className="mt-2 text-sm text-muted-foreground">
                  {customizerProps.defaults?.tagline || customizerProps.defaults?.Tagline || "Tagline goes here"}
                </p>
              </div>
            </div>

            <div className="space-y-4">
              {/* Controls panel placeholder – hook up to your builder */}
              <div className="border rounded-lg p-4">
                <h3 className="font-medium mb-2">Controls</h3>
                <p className="text-sm text-muted-foreground">
                  This panel will host the sliders/inputs (size, rotation, curve, colors, title, tagline) from your customizer.
                </p>
              </div>
            </div>
          </div>
        </>
      )}
    </div>
  );
}