import { useState, useEffect } from "react";
import { useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";

const PATHS: string[] = [
  // Public pages
  "/",
  "/about",
  "/services",
  "/products",
  "/terms",
  "/privacy",
  "/resources",
  "/faq",
  
  // Pricing pages
  "/pricing",
  "/pricing/business-development",
  "/pricing/web-services",
  "/pricing/addons",
  
  // Business Development (corrected paths)
  "/business-development/brand-name-wizard",
  "/business-development/slogan-generator",
  "/business-development/ai-business-plan-developer",
  "/business-development/ai-logo-creator",
  
  // Brand Development
  "/brand-development/ai-logo-creator/logo-composer-coffee-cup-v1",
  "/brand-development/brand-kit-generator",
  "/brand-development/brand-voice-generator",
  
  // Business Assets
  "/business-assets/logo-templates",
  "/business-assets/templates/business-plan",
  "/business-assets/templates/presentation-template",
  
  // Social & Web (corrected paths)
  "/social-kits",
  "/social-content/infographics-library",
  "/website-builder",
  "/domains",
  
  // Marketplace
  "/marketplace",
  
  // Dashboard & Account (corrected paths)
  "/dashboard",
  "/settings",
  "/profile",
  "/account/purchases"
];

interface AuditResult {
  path: string;
  status: "ok" | "not-found";
}

export default function LinkAuditPage() {
  const [, nav] = useLocation();
  const [running, setRunning] = useState(false);
  const [results, setResults] = useState<AuditResult[]>([]);

  // Security: Block access in production
  useEffect(() => {
    if (import.meta.env.MODE === 'production') {
      nav('/', { replace: true });
    }
  }, [nav]);

  const run = async () => {
    setRunning(true);
    const out: AuditResult[] = [];
    
    for (const p of PATHS) {
      // Navigate client-side
      nav(p, { replace: true });
      
      // Wait for page to render
      // eslint-disable-next-line no-await-in-loop
      await new Promise(r => setTimeout(r, 350));
      
      // Check for NotFound page marker
      const notFound = !!document.querySelector("[data-not-found]");
      out.push({ path: p, status: notFound ? "not-found" : "ok" });
    }
    
    setResults(out);
    setRunning(false);
    
    // Navigate back to audit page
    nav("/__qa/link-audit", { replace: true });
  };

  const exportJson = () => {
    const blob = new Blob(
      [JSON.stringify({ at: new Date().toISOString(), items: results }, null, 2)],
      { type: "application/json" }
    );
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = "spa-link-report.json";
    a.click();
    URL.revokeObjectURL(url);
  };

  const broken = results.filter(r => r.status === "not-found");
  const passed = results.filter(r => r.status === "ok");

  return (
    <DashboardTemplatePage title="Link Audit">
      <Card>
        <CardHeader>
          <CardTitle className="text-2xl font-bold">SPA Link Audit (In-App)</CardTitle>
          <CardDescription>
            Navigates key routes client-side and detects the NotFound page via{" "}
            <code className="bg-orange-500 text-white px-1 py-0.5 rounded text-xs">[data-not-found]</code> attribute.
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="flex gap-2 mb-6">
            <Button
              onClick={run}
              disabled={running}
              data-testid="button-run-audit"
            >
              {running ? "Running..." : "Run Audit"}
            </Button>
            <Button
              variant="outline"
              onClick={exportJson}
              disabled={!results.length}
              data-testid="button-export-json"
            >
              Export JSON
            </Button>
          </div>

          {!!results.length && (
            <div>
              <div className="mb-4 p-4 bg-muted rounded-lg">
                <div className="grid grid-cols-3 gap-4 text-center">
                  <div>
                    <div className="text-2xl font-bold">{results.length}</div>
                    <div className="text-sm text-muted-foreground">Total Checked</div>
                  </div>
                  <div>
                    <div className="text-2xl font-bold text-green-600 dark:text-green-400">
                      {passed.length}
                    </div>
                    <div className="text-sm text-muted-foreground">Passed</div>
                  </div>
                  <div>
                    <div className="text-2xl font-bold text-red-600 dark:text-red-400">
                      {broken.length}
                    </div>
                    <div className="text-sm text-muted-foreground">404s</div>
                  </div>
                </div>
              </div>

              <div className="space-y-1">
                {results.map(r => (
                  <div
                    key={r.path}
                    className={`px-3 py-2 rounded text-sm font-mono ${
                      r.status === "not-found"
                        ? "bg-red-50 dark:bg-red-950 text-red-700 dark:text-red-300"
                        : "bg-green-50 dark:bg-green-950 text-green-700 dark:text-green-300"
                    }`}
                    data-testid={`result-${r.status}-${r.path}`}
                  >
                    <span className="mr-2">
                      {r.status === "not-found" ? "❌" : "✅"}
                    </span>
                    {r.path}
                  </div>
                ))}
              </div>
            </div>
          )}

          {!results.length && !running && (
            <div className="text-center py-8 text-muted-foreground">
              <p>Click "Run Audit" to test {PATHS.length} internal routes for 404 errors.</p>
            </div>
          )}
        </CardContent>
      </Card>
    </DashboardTemplatePage>
  );
}
