import { useState, useEffect } from "react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { useLocation } from "wouter";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { useAuth } from "@/contexts/AuthContext";

type Health = {
  ok: boolean;
  env: string;
  hostname: string;
  uptimeSec: number;
  corsOrigins: string[];
  opensrsConfigured: boolean;
  opensrsMode?: string | null;
  domainProvider?: string;
  whmcsUrlHost?: string | null;
  domainUrl?: string | null;
  time: string;
};

type TelemetryEvent = {
  t: string;
  path?: string;
  from?: string;
  ts: number;
  meta?: any;
};

type DomainPing = {
  ok: boolean;
  provider: string;
  configured: boolean;
  latencyMs?: number;
  mode?: string;
  sample?: string;
  message?: string;
  available?: boolean;
};

type StripePing = {
  ok: boolean;
  configured: boolean;
  stripeKeyPresent?: boolean;
  domainUrl?: string | null;
  message?: string;
  checks?: Array<{
    url: string;
    ok: boolean;
    status?: number;
    latencyMs?: number;
    error?: string;
  }>;
};

export default function DiagnosticsPage() {
  const [, nav] = useLocation();
  const { currentUser } = useAuth();
  const [health, setHealth] = useState<Health | null>(null);
  const [events, setEvents] = useState<TelemetryEvent[]>([]);
  const [loading, setLoading] = useState(false);
  const [pingLoading, setPingLoading] = useState(false);
  const [ping, setPing] = useState<DomainPing | null>(null);
  const [stripePingLoading, setStripePingLoading] = useState(false);
  const [stripePing, setStripePing] = useState<StripePing | null>(null);
  const [authorized, setAuthorized] = useState<boolean | null>(null);

  // Note: Owner authorization check below handles security
  // Production access is allowed for owners only

  // Check user authorization (owner or management only)
  useEffect(() => {
    const checkAuthorization = async () => {
      if (!currentUser) {
        setAuthorized(false);
        nav('/dashboard', { replace: true });
        return;
      }

      try {
        const token = await currentUser.getIdToken();
        const response = await fetch("/api/auth/me", {
          headers: { Authorization: `Bearer ${token}` },
          credentials: "include",
        });

        if (response.ok) {
          const userData = await response.json();
          
          // Only allow owners (management role check would go here if needed)
          if (userData.isOwner) {
            setAuthorized(true);
          } else {
            setAuthorized(false);
            nav('/dashboard', { replace: true });
          }
        } else {
          setAuthorized(false);
          nav('/dashboard', { replace: true });
        }
      } catch (err) {
        console.error("Failed to check authorization:", err);
        setAuthorized(false);
        nav('/dashboard', { replace: true });
      }
    };

    checkAuthorization();
  }, [currentUser, nav]);

  const load = async () => {
    setLoading(true);
    try {
      const [healthRes, eventsRes] = await Promise.all([
        fetch("/api/health").then(r => r.json()),
        fetch("/api/telemetry/recent").then(r => r.json())
      ]);
      setHealth(healthRes);
      setEvents(eventsRes?.items || []);
    } catch (error) {
      console.error('Failed to load diagnostics:', error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    load();
  }, []);

  const broken = events.filter(e => e.t === "spa-404");
  const redirects = events.filter(e => e.t === "redirect");

  // Show loading state while checking authorization
  if (authorized === null) {
    return (
      <DashboardTemplatePage title="Diagnostics">
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin h-8 w-8 border-2 border-primary border-t-transparent rounded-full"></div>
        </div>
      </DashboardTemplatePage>
    );
  }

  // Don't render content if not authorized (will redirect)
  if (!authorized) {
    return null;
  }

  return (
    <DashboardTemplatePage title="Diagnostics">
      <div className="mb-6">
        <p className="text-sm text-muted-foreground">
          Runtime health & recent telemetry events (Owner Only)
        </p>
      </div>

      <div className="grid md:grid-cols-2 gap-4 mb-4">
        {/* Server Health */}
        <Card className="p-4">
          <h2 className="font-semibold mb-3 flex items-center gap-2">
            <span className="text-green-600">●</span> Server Health
          </h2>
          {!health ? (
            <div className="text-sm text-muted-foreground">Loading…</div>
          ) : (
            <ul className="text-sm space-y-1.5">
              <li>
                <b>Status:</b>{" "}
                <span className={health.ok ? "text-green-600" : "text-red-600"}>
                  {health.ok ? "OK" : "Down"}
                </span>
              </li>
              <li><b>Environment:</b> {health.env}</li>
              <li><b>Hostname:</b> <code className="text-xs bg-orange-500 text-white px-1 py-0.5 rounded">{health.hostname}</code></li>
              <li><b>Uptime:</b> {Math.floor(health.uptimeSec / 60)}m {health.uptimeSec % 60}s</li>
              <li>
                <b>Domain provider:</b>{" "}
                <span className="bg-orange-500 text-white px-2 py-0.5 rounded font-mono text-sm">
                  {health.domainProvider || "opensrs"}
                </span>
                {health.whmcsUrlHost && (
                  <span className="ml-2 text-gray-600">
                    → {health.whmcsUrlHost}
                  </span>
                )}
              </li>
              <li>
                <b>OpenSRS:</b>{" "}
                {health.opensrsConfigured ? (
                  <span className="text-green-600">
                    configured ({health.opensrsMode || "?"})
                  </span>
                ) : (
                  <span className="text-yellow-600">missing</span>
                )}
              </li>
              <li>
                <b>CORS Origins:</b>{" "}
                {health.corsOrigins?.length ? (
                  <code className="text-xs bg-orange-500 text-white px-1 py-0.5 rounded">
                    {health.corsOrigins.join(", ")}
                  </code>
                ) : (
                  <span className="text-muted-foreground">none (allow all)</span>
                )}
              </li>
              <li>
                <b>Domain URL:</b>{" "}
                {health.domainUrl ? (
                  <code className="text-xs bg-orange-500 text-white px-1 py-0.5 rounded">{health.domainUrl}</code>
                ) : (
                  <span className="text-yellow-600">not set</span>
                )}
              </li>
              <li className="text-xs text-muted-foreground pt-2">
                Last checked: {new Date(health.time).toLocaleString()}
              </li>
            </ul>
          )}
          
          {/* Ping Domains Provider & Stripe URLs */}
          <div className="mt-3 flex items-center gap-2 flex-wrap">
            <Button
              variant="outline"
              size="sm"
              onClick={async () => {
                setPingLoading(true);
                try {
                  const res = await fetch("/api/domain/ping");
                  const data = await res.json();
                  setPing(data);
                } finally {
                  setPingLoading(false);
                }
              }}
              disabled={pingLoading}
              data-testid="button-ping-domains"
            >
              {pingLoading ? "Pinging…" : "Ping Domains Provider"}
            </Button>
            {ping && (
              <span className={`text-xs ${ping.ok ? "text-emerald-600" : "text-red-600"} font-medium`}>
                Domains: {ping.configured ? (
                  ping.ok ? (
                    <>OK {ping.latencyMs}ms</>
                  ) : (
                    <>Failed {ping.latencyMs ?? "-"}ms</>
                  )
                ) : (
                  "Not configured"
                )}
                {ping.mode && <> • {ping.mode}</>}
                {ping.sample && <> • {ping.sample}</>}
                {ping.message && <> • {ping.message}</>}
              </span>
            )}
            
            <Button
              variant="outline"
              size="sm"
              onClick={async () => {
                setStripePingLoading(true);
                try {
                  const res = await fetch("/api/stripe/ping");
                  const data = await res.json();
                  setStripePing(data);
                } finally {
                  setStripePingLoading(false);
                }
              }}
              disabled={stripePingLoading}
              data-testid="button-ping-stripe"
            >
              {stripePingLoading ? "Pinging…" : "Ping Stripe URLs"}
            </Button>
            {stripePing && (
              <span className={`text-xs ${stripePing.ok ? "text-emerald-600" : "text-red-600"} font-medium`}>
                Stripe: {stripePing.configured ? (stripePing.ok ? "OK" : "Issues") : "Not configured"}
                {stripePing.domainUrl && <> • {stripePing.domainUrl}</>}
              </span>
            )}
          </div>
          
          {/* Stripe URL Check Details */}
          {stripePing?.checks?.length ? (
            <div className="mt-2 text-xs text-muted-foreground space-y-1">
              {stripePing.checks.map((c, i) => (
                <div key={i} className="font-mono">
                  {c.ok ? "✅" : "❌"} {c.url} — {c.latencyMs ?? "-"}ms{" "}
                  {c.status ? `(HTTP ${c.status})` : (c.error ? `(${c.error})` : "")}
                </div>
              ))}
            </div>
          ) : null}
        </Card>

        {/* Recent 404s */}
        <Card className="p-4">
          <div className="flex items-center justify-between mb-3">
            <h2 className="font-semibold">Recent 404s</h2>
            <span className="text-xs bg-orange-500 text-white px-2 py-1 rounded">
              {broken.length} events
            </span>
          </div>
          <div className="max-h-64 overflow-auto">
            {!broken.length ? (
              <div className="text-sm text-muted-foreground">No 404s recorded</div>
            ) : (
              <div className="space-y-2">
                {broken.map((e, i) => (
                  <div
                    key={i}
                    className="text-sm py-2 px-3 bg-orange-500 text-white rounded"
                  >
                    <div className="font-mono text-xs">
                      ❌ <b>{e.path}</b>
                    </div>
                    <div className="text-xs text-white/80 mt-0.5">
                      {new Date(e.ts).toLocaleString()}
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>
        </Card>

        {/* Recent Redirects */}
        <Card className="p-4 md:col-span-2">
          <div className="flex items-center justify-between mb-3">
            <h2 className="font-semibold">Recent Redirects</h2>
            <div className="flex items-center gap-2">
              <span className="text-xs bg-orange-500 text-white px-2 py-1 rounded">
                {redirects.length} events
              </span>
              <Button
                variant="outline"
                size="sm"
                onClick={load}
                disabled={loading}
                data-testid="button-refresh"
              >
                {loading ? "Refreshing..." : "Refresh"}
              </Button>
            </div>
          </div>
          <div className="max-h-64 overflow-auto">
            {!redirects.length ? (
              <div className="text-sm text-muted-foreground">No redirects recorded</div>
            ) : (
              <div className="space-y-2">
                {redirects.map((e, i) => (
                  <div
                    key={i}
                    className="text-sm py-2 px-3 bg-orange-500 text-white rounded"
                  >
                    <div className="font-mono text-xs">
                      ↪ <span className="text-white/80">{e.from}</span> →{" "}
                      <b>{e.path}</b>
                    </div>
                    <div className="text-xs text-white/80 mt-0.5">
                      {new Date(e.ts).toLocaleString()}
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>
        </Card>
      </div>

      <div className="flex gap-2">
        <Button onClick={load} disabled={loading} data-testid="button-reload">
          {loading ? "Reloading…" : "Reload Data"}
        </Button>
        <Button variant="outline" onClick={() => nav("/__qa/link-audit")}>
          Link Audit Tool →
        </Button>
      </div>
    </DashboardTemplatePage>
  );
}
