import { useState, useEffect } from "react";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { TemplateCard } from "@/components/templates/TemplateCard";
import { type FirebaseLogoTemplate } from "@/services/firebase-templates";
import { Loader2, Search, ChevronRight } from "lucide-react";
import { Input } from "@/components/ui/input";
import { Breadcrumb, BreadcrumbList, BreadcrumbItem, BreadcrumbLink, BreadcrumbPage, BreadcrumbSeparator } from "@/components/ui/breadcrumb";

export default function LogoTemplatesPage() {
  const [items, setItems] = useState<FirebaseLogoTemplate[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");

  useEffect(() => {
    (async () => {
      try {
        setLoading(true);
        // Use backend API instead of direct Firestore access
        const response = await fetch('/api/logo-templates-firebase/list');
        if (!response.ok) throw new Error(`HTTP ${response.status}`);
        const data = await response.json();
        setItems(data.items || []);
      } catch (err: any) {
        console.error("Failed to load logo templates:", err);
      } finally {
        setLoading(false);
      }
    })();
  }, []);

  const filtered = items.filter((t) =>
    [t.name, t.description, t.id, ...(t.tags || [])]
      .filter(Boolean)
      .some((val) => val.toLowerCase().includes(searchTerm.toLowerCase()))
  );

  return (
    <DashboardTemplatePage title="Logo Templates">
      {/* Breadcrumbs */}
      <Breadcrumb className="mb-[16px] -mt-1">
        <BreadcrumbList>
          <BreadcrumbItem>
            <BreadcrumbLink href="/dashboard" className="text-[#FF8B00] hover:text-[#e67d00]">Dashboard</BreadcrumbLink>
          </BreadcrumbItem>
          <BreadcrumbSeparator>
            <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
          </BreadcrumbSeparator>
          <BreadcrumbItem>
            <BreadcrumbLink className="text-[#FF8B00] hover:text-[#e67d00]">Business Assets</BreadcrumbLink>
          </BreadcrumbItem>
          <BreadcrumbSeparator>
            <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
          </BreadcrumbSeparator>
          <BreadcrumbItem>
            <BreadcrumbPage className="text-[#FF8B00] font-semibold">Logo Templates</BreadcrumbPage>
          </BreadcrumbItem>
        </BreadcrumbList>
      </Breadcrumb>

      <div className="space-y-6">
        {/* Step 1 Header */}
        <div className="mb-6">
          <h2 className="text-xl font-semibold mb-2 text-gray-900 dark:text-gray-100">Step 1: Select a Logo</h2>
          <p className="text-sm text-gray-600 dark:text-gray-400">
            Choose a template. You can <span className="font-medium">download the SVG for free</span>,
            or <span className="font-medium">become a member to open it in the Logo Composer</span> and add
            custom text, shapes, and colors.
          </p>
        </div>

        {/* Search Bar */}
        <div className="relative max-w-md">
          <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-muted-foreground w-4 h-4" />
          <Input
            placeholder="Search templates..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            className="pl-9"
            data-testid="input-search-templates"
          />
        </div>

        {loading ? (
          <LoadingState />
        ) : filtered.length === 0 ? (
          <EmptyState />
        ) : (
          <>
            <div className="flex items-center justify-between">
              <p className="text-sm text-muted-foreground">
                {filtered.length} template{filtered.length !== 1 ? "s" : ""} available
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
              {filtered.map((t) => {
                const previewUrl = t.assets.previewDownloadUrl || t.assets.svgDownloadUrl;
                const type = t.assets.svgDownloadUrl ? "SVG" : "—";
                return (
                  <TemplateCard
                    key={t.id}
                    id={t.id}
                    name={t.name}
                    description={t.description}
                    tags={[...(t.tags || []), t.id]}
                    previewUrl={previewUrl}
                    source="firebase"
                    typeOverride={type}
                    defaults={{
                      brandName: t.defaults?.brandName,
                      tagline: t.defaults?.tagline,
                      estYear: t.defaults?.estYear,
                    }}
                  />
                );
              })}
            </div>

          </>
        )}
      </div>
    </DashboardTemplatePage>
  );
}

function LoadingState() {
  return (
    <div className="flex items-center justify-center py-12">
      <div className="text-center">
        <Loader2 className="w-8 h-8 animate-spin mx-auto mb-4 text-muted-foreground" />
        <p className="text-muted-foreground">Loading templates...</p>
      </div>
    </div>
  );
}

function EmptyState() {
  return (
    <div className="text-center py-12">
      <p className="text-muted-foreground mb-2">No templates available yet</p>
      <p className="text-sm text-gray-500">Upload some logo templates from the admin panel to get started!</p>
    </div>
  );
}