import React, { useEffect, useMemo, useState } from "react";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Dialog, DialogContent, DialogTitle } from "@/components/ui/dialog";
import { Download, Search, X, ShoppingCart } from "lucide-react";
import { useCart } from "@/contexts/CartContext";
import { useToast } from "@/hooks/use-toast";
import { useEntitlement } from "@/hooks/useEntitlement";
import { fileTypeFrom, toTenDigitCode } from "@/utils/assetDisplay";
import StockCard from "../../../components/StockCard";

type Item = { 
  id: string; 
  url: string; 
  name: string; 
  tags: string[]; 
  category?: string; 
  createdAt: number;
  mimeType?: string;
  code?: string;
};

export default function StockPhotos() {
  const [data, setData] = useState<Item[]>([]);
  const [q, setQ] = useState("");
  const [loading, setLoading] = useState(true);
  const [selectedImage, setSelectedImage] = useState<Item | null>(null);
  const { addItem, open } = useCart();
  const { toast } = useToast();
  const [selectedImageCode, setSelectedImageCode] = useState("");
  const ent = useEntitlement(selectedImage?.id);


  const STOCK_PHOTO_PRICE = 699; // $6.99 in cents

  const handleAddToCart = async (item: Item) => {
    const code = item.code || await toTenDigitCode(item.id);
    const displayName = `IBrandBiz | #${code}`;
    
    addItem({
      kind: "stock",
      assetId: item.id,
      name: displayName,
      priceCents: STOCK_PHOTO_PRICE,
      qty: 1,
      previewUrl: `/api/stock/${item.id}/preview`,
    });
    open();

    toast({
      title: "Added to cart",
      description: `${displayName} added to your cart`,
    });
  };

  const handleDownload = () => {
    if (selectedImage) {
      window.location.href = `/api/stock/${selectedImage.id}/download`;
    }
  };

  const handlePurchase = async () => {
    if (selectedImage) {
      await handleAddToCart(selectedImage);
    }
  };

  // Update selected image code when image changes
  useEffect(() => {
    if (selectedImage) {
      (async () => {
        const code = selectedImage.code || await toTenDigitCode(selectedImage.id);
        setSelectedImageCode(code);
      })();
    }
  }, [selectedImage]);


  // Handle keyboard events for modal
  useEffect(() => {
    const handleKeyDown = (event: KeyboardEvent) => {
      if (event.key === 'Escape' && selectedImage) {
        setSelectedImage(null);
      }
    };

    if (selectedImage) {
      document.addEventListener('keydown', handleKeyDown);
      document.body.style.overflow = 'hidden'; // Prevent background scrolling
    }

    return () => {
      document.removeEventListener('keydown', handleKeyDown);
      document.body.style.overflow = 'unset';
    };
  }, [selectedImage]);

  useEffect(() => { 
    (async () => {
      try {
        const res = await fetch("/api/stock/photos");
        const result = await res.json();
        // Handle both direct array and object with photos property
        setData(Array.isArray(result) ? result : (result.photos || []));
      } catch (error) {
        console.error("Failed to fetch stock photos:", error);
      } finally {
        setLoading(false);
      }
    })(); 
  }, []);

  const filtered = useMemo(() => {
    const term = q.toLowerCase().trim();
    if (!term) return data;
    return data.filter(x =>
      x.name.toLowerCase().includes(term) ||
      x.tags.join(" ").toLowerCase().includes(term) ||
      (x.category||"").toLowerCase().includes(term)
    );
  }, [data, q]);

  return (
    <DashboardTemplatePage title="Stock Photos" description="High-quality stock photos for your business and marketing needs">
      <div className="space-y-6">
        {/* Search Section */}
        <div className="flex items-center gap-4">
          <div className="relative flex-1 max-w-md">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground h-4 w-4" />
            <Input
              value={q}
              onChange={(e) => setQ(e.target.value)}
              placeholder="Search by name, tags, or category..."
              className="pl-10"
              data-testid="input-search"
            />
          </div>
          <div className="text-sm text-muted-foreground">
            {filtered.length} {filtered.length === 1 ? 'photo' : 'photos'} found
          </div>
        </div>

        {/* Loading State */}
        {loading && (
          <div className="flex items-center justify-center py-12">
            <div className="animate-spin h-8 w-8 border-2 border-primary border-t-transparent rounded-full"></div>
          </div>
        )}

        {/* Empty State */}
        {!loading && data.length === 0 && (
          <div className="text-center py-12">
            <div className="text-muted-foreground mb-4">
              <svg className="h-16 w-16 mx-auto mb-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
              </svg>
            </div>
            <h3 className="text-lg font-medium mb-2">No stock photos available</h3>
            <p className="text-muted-foreground">Stock photos will appear here once uploaded by administrators.</p>
          </div>
        )}

        {/* Photos Grid */}
        {!loading && filtered.length > 0 && (
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
            {filtered.map((item) => {
              // Convert item to asset format with proper preview URL
              const asset = {
                id: item.id,
                name: item.name,
                previewUrl: `/api/stock/${item.id}/preview`,
                mimeType: item.mimeType,
                code: item.code,
                tags: item.tags,
                category: item.category
              };
              
              return (
                <StockCard 
                  key={item.id} 
                  asset={asset} 
                  onImageClick={setSelectedImage}
                />
              );
            })}
          </div>
        )}

        {/* No Results State */}
        {!loading && data.length > 0 && filtered.length === 0 && (
          <div className="text-center py-12">
            <div className="text-muted-foreground mb-4">
              <Search className="h-16 w-16 mx-auto mb-4" />
            </div>
            <h3 className="text-lg font-medium mb-2">No photos found</h3>
            <p className="text-muted-foreground">Try adjusting your search terms or browse all photos.</p>
          </div>
        )}
      </div>

      {/* Image Modal/Lightbox */}
      <Dialog open={!!selectedImage} onOpenChange={() => setSelectedImage(null)}>
        <DialogContent className="max-w-6xl w-[95vw] max-h-[95vh] p-0 bg-transparent border-none shadow-none" data-testid="modal-image-preview">
          <DialogTitle className="sr-only">Stock Photo Preview</DialogTitle>
          <div className="relative bg-black/90 rounded-lg overflow-hidden h-[90vh] flex flex-col">
            {selectedImage && (
              <>
                {/* Close Button */}
                <button
                  onClick={() => setSelectedImage(null)}
                  className="absolute top-4 right-4 z-10 bg-black/50 hover:bg-black/70 text-white rounded-full p-2 transition-colors"
                  data-testid="modal-close-button"
                >
                  <X className="h-5 w-5" />
                </button>
                <div className="flex items-center justify-center flex-1 p-4 min-h-0">
                  <img 
                    src={selectedImage.previewUrl || selectedImage.url} 
                    alt={selectedImage.name}
                    className="max-h-full max-w-full w-auto h-auto mx-auto object-contain rounded"
                  />
                </div>
                <div className="bg-black/70 text-white p-4 flex-shrink-0">
                  <h3 className="font-semibold text-lg mb-1">IBrandBiz | #{selectedImageCode}</h3>
                  <p className="text-xs text-gray-300 mb-2">File type: {fileTypeFrom(selectedImage)}</p>
                  <div className="flex flex-wrap gap-2 mb-3">
                    {selectedImage.category && (
                      <Badge variant="secondary" className="text-xs bg-white/20 text-white border-white/30">
                        {selectedImage.category}
                      </Badge>
                    )}
                    {selectedImage.tags?.slice(0, 5).map((tag, index) => (
                      <Badge key={index} variant="outline" className="text-xs border-white/30 text-white">
                        {tag}
                      </Badge>
                    ))}
                    {selectedImage.tags && selectedImage.tags.length > 5 && (
                      <Badge variant="outline" className="text-xs border-white/30 text-white">
                        +{selectedImage.tags?.length ? selectedImage.tags.length - 5 : 0}
                      </Badge>
                    )}
                  </div>
                  <div className="text-xs text-gray-300 mb-3">{selectedImage.name}</div>
                  <div className="flex gap-3">
                    {(() => {
                      if (ent.loading) return (
                        <Button disabled className="bg-gray-600 text-white">
                          Checking...
                        </Button>
                      );
                      if (ent.licensed) return (
                        <Button
                          onClick={handleDownload}
                          className="bg-green-600 hover:bg-green-700 text-white"
                          data-testid={`modal-download-${selectedImage.id}`}
                        >
                          <Download className="h-4 w-4 mr-2" />
                          Download
                        </Button>
                      );
                      if (ent.quotaRemaining > 0) return (
                        <Button
                          onClick={handleDownload}
                          className="bg-green-600 hover:bg-green-700 text-white"
                          data-testid={`modal-download-credit-${selectedImage.id}`}
                        >
                          <Download className="h-4 w-4 mr-2" />
                          Download (uses 1 credit)
                        </Button>
                      );
                      return (
                        <Button
                          onClick={handlePurchase}
                          className="bg-blue-600 hover:bg-blue-700 text-white"
                          data-testid={`modal-purchase-${selectedImage.id}`}
                        >
                          <ShoppingCart className="h-4 w-4 mr-2" />
                          Purchase $6.99
                        </Button>
                      );
                    })()}
                  </div>
                </div>
              </>
            )}
          </div>
        </DialogContent>
      </Dialog>
    </DashboardTemplatePage>
  );
}